import logging
import traceback

from aiohttp import web

from crm.infra.github_webhook.pr_events.ping import process_ping
from crm.infra.github_webhook.pr_events.merged import process_merged_pr
from crm.infra.github_webhook.pr_events.change import process_opened_pr, process_changed_reviewers, process_pr_approved,\
    process_pr_unapproved, process_pr_needs_work

from crm.infra.github_webhook.pr_events.comment import process_added_comment

from crm.infra.github_webhook.utils.signature import check_sha1_signature
from crm.infra.github_webhook.utils.st import get_tickets, execute_transitions


HANDLERS = {
    'diagnostics:ping': process_ping,
    'pr:merged': process_merged_pr,
    'pr:opened': process_opened_pr,
    'pr:reviewer:updated': process_changed_reviewers,
    'pr:reviewer:approved': process_pr_approved,
    'pr:reviewer:unapproved': process_pr_unapproved,
    'pr:reviewer:needs_work': process_pr_needs_work,
    'pr:comment:added': process_added_comment
}

logger = logging.getLogger(__name__)


async def handle_bb_webhook(request):
    try:
        event_key = request.headers['X-Event-Key']
        logger.info('processing {} event key'.format(event_key))
        if event_key not in HANDLERS:
            logger.warn('event key unknown')
            return web.Response(status=400, text='unknown event_key')

        handler = HANDLERS[event_key]
        logger.debug('use {} handler'.format(handler))

        return await handler(request)
    except Exception as ex:
        logger.exception(ex)
        return web.Response(status=500, text=traceback.format_exc())


async def handle_github_webhook(request):
    if not await check_sha1_signature(request):
        return web.Response(status=401)

    body = await request.json()
    if not body['action'] == 'closed' or not body['pull_request']['merged']:
        return web.Response(status=200)

    tickets = get_tickets(body['pull_request']['title'], body['pull_request']['body'], '')
    errors = await execute_transitions(tickets)
    if not errors:
        errors = web.Response(status=200)
    return errors
