package ru.yandex.crm.infra.logbroker.hoover

import com.google.protobuf.Empty
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery
import com.yandex.ydb.persqueue.v1.ClusterDiscoveryServiceGrpc
import io.grpc.ManagedChannelBuilder
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.crm.infra.logbroker.hoover.config.ApiConfig
import ru.yandex.crm.infra.logbroker.hoover.config.HooverConfig
import ru.yandex.crm.infra.logbroker.hoover.config.LogbrokerConfig
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory
import ru.yandex.kikimr.persqueue.auth.Credentials
import ru.yandex.kikimr.persqueue.consumer.StreamConsumer
import ru.yandex.kikimr.persqueue.consumer.stream.StreamConsumerConfig
import ru.yandex.kikimr.persqueue.ydb.YdbCoreConfig
import ru.yandex.passport.tvmauth.NativeTvmClient
import ru.yandex.passport.tvmauth.TvmClient
import ru.yandex.passport.tvmauth.TvmToolSettings
import java.lang.Exception

class HooverBridge {
    private val logger = logger()

    fun start() {
        logger.info("Process started!")

        val config = HooverConfig.fromEnvironmentVariables()
        logger.info("Hoover config: {}", config)

        val lbTvmClient = NativeTvmClient(TvmToolSettings.create(config.logbroker.tvm.selfAlias))
        val apiTvmClient = if (config.logbroker.tvm.selfAlias == config.api.tvm.selfAlias)
            lbTvmClient
        else
            NativeTvmClient(TvmToolSettings.create(config.api.tvm.selfAlias))

        val consumerConfig = createConsumerConfig(config.logbroker, lbTvmClient)
        val listenerBuilder = createListenerBuilder(config.api, apiTvmClient)

        val clusters = getClusters(config.logbroker)
        val consumers = ArrayList<Pair<StreamConsumer, Listener>>(clusters.size)
        for (cluster in clusters.filter { c -> c.available }) {
            try {
                val ydbConfig = YdbCoreConfig(cluster.endpoint, config.logbroker.database, false)
                val factory = LogbrokerClientFactory(ydbConfig)
                logger.debug("Creating consumer for {}", cluster.endpoint)

                val listener = listenerBuilder.build()
                val consumer = factory.streamConsumer(consumerConfig)
                consumer.startConsume(listener)
                logger.info("Reading for {} started", cluster.endpoint)
                consumers.add(Pair(consumer, listener))
            }
            catch (e : Exception){
                logger.error("Error while creating consumer for {}", cluster.endpoint, e)
            }
        }
        while (true) {
            // TODO: graceful shutdown with Ya.Deploy stop probe
            Thread.sleep(5000)
            Thread.yield()
        }
    }

    private fun createConsumerConfig(config: LogbrokerConfig, tvmClient: TvmClient): StreamConsumerConfig =
        StreamConsumerConfig.builder(config.topics, config.consumer)
            .configureReader { cb -> cb.setMaxCount(config.maxMessagesCount) }
            .configureSession { cb -> cb.setReadOnlyLocal(true) }
            .configureRetries { cb -> cb.enable() }
            .setCredentialsProvider { Credentials.tvm(tvmClient.getServiceTicketFor(config.tvm.destAlias)) }
            .build()

    private fun createListenerBuilder(apiConfig: ApiConfig, tvmClient: TvmClient): ListenerBuilder =
        ListenerBuilder(apiConfig, tvmClient)

    private fun getClusters(config: LogbrokerConfig): List<YdbPersqueueClusterDiscovery.ClusterInfo> {
        val channel = ManagedChannelBuilder.forTarget(config.installation).usePlaintext().build()

        val readSessionsBuilder = YdbPersqueueClusterDiscovery.ReadSessionParams.newBuilder()
            .setAllOriginal(Empty.newBuilder())
        val discoverClustersRequestBuilder = YdbPersqueueClusterDiscovery.DiscoverClustersRequest.newBuilder()
        config.topics.forEach {
            readSessionsBuilder.topic = it
            discoverClustersRequestBuilder.addReadSessions(readSessionsBuilder)
        }

        val discoveryResponse = ClusterDiscoveryServiceGrpc.newBlockingStub(channel)
            .discoverClusters(discoverClustersRequestBuilder.build())
        val discoveryResult = YdbPersqueueClusterDiscovery.DiscoverClustersResult
            .parseFrom(discoveryResponse.operation.result.value)

        channel.shutdown()

        logger.info("Discovery result {}", discoveryResult)
        return discoveryResult.readSessionsClustersList[0].clustersList
    }
}

fun <T : Any> T.logger(): Logger {
    return LoggerFactory.getLogger(this.javaClass.name)
}
