package ru.yandex.crm.infra.logbroker.hoover

import com.google.gson.Gson
import com.google.gson.annotations.SerializedName
import com.google.protobuf.CodedOutputStream
import ru.yandex.crm.infra.logbroker.hoover.config.ApiConfig
import ru.yandex.kikimr.persqueue.consumer.StreamListener
import ru.yandex.kikimr.persqueue.consumer.transport.message.CommitMessage
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerInitResponse
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData
import ru.yandex.passport.tvmauth.TvmClient
import java.io.ByteArrayOutputStream
import java.net.HttpURLConnection
import java.net.URL
import java.time.LocalDateTime
import java.time.ZoneOffset

class Listener(
    private val apiConfig: ApiConfig,
    private val tvmClient: TvmClient,
) : StreamListener {
    private val logger = logger()
    private val messagesToSend = mutableListOf<MessageData>()

    data class FailedEventsContract(
        @SerializedName("failedEvents", alternate = ["FailedEvents", "failed_events"]) val FailedEvents: IntArray?
    )

    override fun onInit(init: ConsumerInitResponse?) {
        logger.debug("Consumer init: session={}", init?.sessionId)
    }

    override fun onRead(read: ConsumerReadResponse?, readResponder: StreamListener.ReadResponder?) {
        logger.debug("Consumer read LB message: cookie={}, batches={}", read?.cookie, read?.batches?.size)

        try {
            if (read != null && read.batches != null) {
                val initial = read.batches.joinToString("\n") {
                    "${it.topic} : ${it.partition} : [" +
                        it.messageData.joinToString(", ") { m -> m.offset.toString() } +
                        "]"
                }
                val sorted = read.batches.sortedWith(compareBy({ it.topic }, { it.partition }))
                    .joinToString("\n") { b ->
                        "${b.topic} : ${b.partition} : [" +
                            b.messageData.sortedBy { it.offset }.joinToString(", ") { it.offset.toString() } +
                            "]"
                    }

                messagesToSend.addAll(read.batches.sortedWith(compareBy({ it.topic }, { it.partition })).flatMap { b -> b.messageData.sortedBy { it.offset } })

                logger.info("sorted and initial messages are the same: {}\ninitial:\n{}\nsorted:\n{}", initial == sorted, initial, sorted)
                logger.info("{} messages are ready for send", messagesToSend.count())
                val messagesWithErrors = mutableListOf<MessageData>()

                if (apiConfig.sendBatches) {
                    val byteStream = ByteArrayOutputStream()
                    val codedStream = CodedOutputStream.newInstance(byteStream)
                    val messagesBatch = mutableListOf<MessageData>()
                    for (data in messagesToSend) {
                        codedStream.writeUInt32NoTag(data.decompressedData.size)
                        codedStream.writeRawBytes(data.decompressedData)
                        codedStream.flush()
                        messagesBatch.add(data)
                        if (byteStream.size() > apiConfig.batchMaxSize) {
                            val (success, errors) = sendPostRequest(byteStream.toByteArray())
                            if (success) {
                                for (number in errors) {
                                    messagesWithErrors.add(messagesBatch[number])
                                }
                            } else {
                                messagesWithErrors.addAll(messagesBatch)
                            }

                            byteStream.reset()
                            messagesBatch.clear()
                        }
                    }
                    val (success, errors) = sendPostRequest(byteStream.toByteArray())
                    if (success) {
                        for (number in errors) {
                            messagesWithErrors.add(messagesBatch[number])
                        }
                    } else {
                        messagesWithErrors.addAll(messagesBatch)
                    }
                } else {
                    for (data in messagesToSend) {
                        val (success, errors) = sendPostRequest(data.decompressedData)
                        if (!success || errors.any()) {
                            messagesWithErrors.add(data)
                        }
                    }
                }

                messagesToSend.clear()
                if (messagesWithErrors.any()) {
                    logger.warn("{} messages failed to send", messagesWithErrors.count())
                    for (message in messagesWithErrors) {
                        val nowSeconds = LocalDateTime.now(ZoneOffset.UTC).toEpochSecond(ZoneOffset.UTC)
                        val messageCreateTimeSeconds = message.messageMeta.createTimeMs / 1000
                        if (!apiConfig.ignoreErrors && nowSeconds < messageCreateTimeSeconds + apiConfig.resendMaxOffset) {
                            logger.debug("message {} failed and will be resend", message.offset)
                            messagesToSend.add(message)
                        } else {
                            logger.warn(
                                "message {} failed and will be dropped, dt={}",
                                message.offset,
                                LocalDateTime.ofEpochSecond(messageCreateTimeSeconds, 0, ZoneOffset.UTC)
                            )
                        }
                    }
                }
            }
        } catch (e: Exception) {
            logger.error("Error while reading and sending", e)
        } finally {
            readResponder?.commit()
        }
    }

    override fun onCommit(commit: CommitMessage?) {
        logger.debug("Consumer commit cookies: {}", commit?.cookies)
    }

    override fun onClose() {
        logger.debug("Consumer closed")
    }

    override fun onError(e: Throwable?) {
        logger.error("Consumer error", e)
    }

    private fun sendPostRequest(data: ByteArray): Pair<Boolean, IntArray> {
        if (logger.isDebugEnabled) {
            try {
                logger.debug("Sending to api: {}", data.decodeToString(0, data.size.coerceAtMost(1000)))
            } catch (ex: Exception) {
                logger.warn("Unable to parse data", ex)
            }
        }

        val ticket = tvmClient.getServiceTicketFor(apiConfig.tvm.destAlias)
        val mURL = URL(apiConfig.url)
        var result = true
        val errors = mutableListOf<Int>()

        try {
            with(mURL.openConnection() as HttpURLConnection) {
                requestMethod = "POST"
                setRequestProperty("X-Ya-Service-Ticket", ticket)
                setRequestProperty("Content-Type", "application/json; utf-8")
                doOutput = true

                outputStream.write(data, 0, data.size)
                outputStream.flush()

                when (responseCode / 100) {
                    2 -> {
                        logger.debug("Response: HTTP {}, {}", responseCode, responseMessage)

                        try {
                            val json = inputStream.bufferedReader().readText()
                            logger.debug("Response body: '{}'", json)
                            val answer = Gson().fromJson(json, FailedEventsContract::class.java)
                            if (answer?.FailedEvents != null && answer.FailedEvents.isNotEmpty()) {
                                errors.addAll(answer.FailedEvents.toTypedArray())
                            }
                        } catch (ex: Exception) {
                            logger.info("Cannot parse answer from API", ex)
                        }
                    }
                    4 -> {
                        logger.error("Response: HTTP {}, {}", responseCode, responseMessage)
                        result = false
                    }
                    else -> {
                        logger.warn("Response: HTTP {}, {}", responseCode, responseMessage)
                        result = false
                    }
                }
            }
        } catch (e: Exception) {
            logger.error("Exception while sending to api", e)
            result = false
        }

        return result to errors.toIntArray()
    }
}
