using System;
using System.Linq;
using BaGet.Core;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Routing;
using Microsoft.Extensions.Logging;
using NuGet.Versioning;

namespace BaGet.Hosting
{
    // TODO: This should validate the "Host" header against known valid values
    public class BaGetUrlGenerator : IUrlGenerator
    {
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly LinkGenerator _linkGenerator;
        private readonly ILogger<BaGetUrlGenerator> _logger;

        public BaGetUrlGenerator(
            IHttpContextAccessor httpContextAccessor,
            LinkGenerator linkGenerator,
            ILogger<BaGetUrlGenerator> logger)
        {
            _httpContextAccessor = httpContextAccessor ?? throw new ArgumentNullException(nameof(httpContextAccessor));
            _linkGenerator = linkGenerator ?? throw new ArgumentNullException(nameof(linkGenerator));
            _logger = logger;
        }

        public string GetServiceIndexUrl()
        {
            return GetLink(
                Routes.IndexRouteName,
                values: null);
        }

        public string GetPackageContentResourceUrl()
        {
            return AbsoluteUrl("v3/package");
        }

        public string GetPackageMetadataResourceUrl()
        {
            return AbsoluteUrl("v3/registration");
        }

        public string GetPackagePublishResourceUrl()
        {
            return GetLink(
                Routes.UploadPackageRouteName,
                values: null);
        }

        public string GetSymbolPublishResourceUrl()
        {
            return GetLink(
                Routes.UploadSymbolRouteName,
                values: null);
        }

        public string GetSearchResourceUrl()
        {
            return GetLink(
                Routes.SearchRouteName,
                values: null);
        }

        public string GetAutocompleteResourceUrl()
        {
            return GetLink(
                Routes.AutocompleteRouteName,
                values: null);
        }

        public string GetRegistrationIndexUrl(string id)
        {
            return GetLink(
                Routes.RegistrationIndexRouteName,
                values: new { Id = id.ToLowerInvariant() });
        }

        public string GetRegistrationPageUrl(string id, NuGetVersion lower, NuGetVersion upper)
        {
            // BaGet does not support paging the registration resource.
            throw new NotImplementedException();
        }

        public string GetRegistrationLeafUrl(string id, NuGetVersion version)
        {
            return GetLink(
                Routes.RegistrationLeafRouteName,
                values: new
                {
                    Id = id.ToLowerInvariant(),
                    Version = version.ToNormalizedString().ToLowerInvariant(),
                });
        }

        public string GetPackageVersionsUrl(string id)
        {
            return GetLink(
                Routes.PackageVersionsRouteName,
                values: new { Id = id.ToLowerInvariant() });
        }

        public string GetPackageDownloadUrl(string id, NuGetVersion version)
        {
            id = id.ToLowerInvariant();
            var versionString = version.ToNormalizedString().ToLowerInvariant();

            return GetLink(
                Routes.PackageDownloadRouteName,
                values: new
                {
                    Id = id,
                    Version = versionString,
                    IdVersion = $"{id}.{versionString}"
                });
        }

        public string GetPackageManifestDownloadUrl(string id, NuGetVersion version)
        {
            id = id.ToLowerInvariant();
            var versionString = version.ToNormalizedString().ToLowerInvariant();

            return GetLink(
                Routes.PackageDownloadRouteName,
                values: new
                {
                    Id = id,
                    Version = versionString,
                    Id2 = id,
                });
        }

        public string GetPackageIconDownloadUrl(string id, NuGetVersion version)
        {
            id = id.ToLowerInvariant();
            var versionString = version.ToNormalizedString().ToLowerInvariant();

            return GetLink(Routes.PackageDownloadIconRouteName,
                values: new
                {
                    Id = id,
                    Version = versionString
                });
        }

        private string GetLink(string route, object values)
        {
            _logger.LogInformation(
                $"GetLink({route}): {_httpContextAccessor.HttpContext.Request.Headers["X-Forwarded-Proto"]} | {_httpContextAccessor.HttpContext.Request.Headers["X-Forwarded-Host"]}");

            return _linkGenerator.GetUriByRouteValues(
                _httpContextAccessor.HttpContext,
                route,
                values : values);
        }

        private string AbsoluteUrl(string relativePath)
        {
            var request = _httpContextAccessor.HttpContext.Request;

            var host = request.Headers["X-Forwarded-Host"].FirstOrDefault();
            var proto = request.Headers["X-Forwarded-Proto"].FirstOrDefault() ?? request.Scheme;
            _logger.LogInformation(
                $"AbsoluteUrl({relativePath}): {_httpContextAccessor.HttpContext.Request.Headers["X-Forwarded-Proto"]} | {_httpContextAccessor.HttpContext.Request.Headers["X-Forwarded-Host"]}");

            if (string.IsNullOrWhiteSpace(host))
                host = request.Host.ToUriComponent();

            return string.Concat(
                proto,
                "://",
                host,
                request.PathBase.ToUriComponent(),
                "/",
                relativePath);
        }
    }
}
