import logging
from collections import namedtuple
import locale

from ..config import Config
from .monitors.elastic_monitor_group import ElasticMonitorGroup
from .monitors.mail import MailMonitor
from .monitors.ora import OraMonitor
from .utils.stopwatch import Stopwatch
from .utils.graphite_client import GraphiteClient
from .utils.mail_client import MailClient
from .utils.ora_client import Oracle


class Pymonitor:

    def __init__(self):
        locale.setlocale(locale.LC_NUMERIC, '')
        self.logger = logging.getLogger(self.__class__.__name__)
        self.monitors = {
            "1min": [],
            "5min": [],
            "20min": [],
            "1hour": [],
            "1day": [],
        }
        self.init_basic_monitors()
        self.init_ora_mons()
        self.dump_monitors()
        self._monitor_results = {}

    @property
    def monitor_results(self):
        return self._monitor_results

    def clean_results(self):
        self._monitor_results = {}

    def init_basic_monitors(self):
        self.monitors["1min"] = [
            MailMonitor("one_min.crm_yandex_ru.mail.tcrm.new", MailClient("tcrm", Config.MAIL_TOKEN_TCRM)),
            MailMonitor("one_min.crm_yandex_ru.mail.tcrm-support.new",
                        MailClient("robot-tcrm-support", Config.MAIL_TOKEN_ROBOT_TCRM)),
        ]

    def get_signal(self):
        return GraphiteClient(Config.GRAPHITE_HOST, Config.GRAPHITE_PORT)

    def stop(self):
        self.logger.info("Stopping")

    def start(self):
        self.logger.info("Initializing")

    def init_ora_mons(self):
        self.logger.debug("Init sql-monitors")
        # removing existed oracle monitors
        for key in self.monitors:
            self.monitors[key] = [mon for mon in self.monitors[key] if not isinstance(mon, OraMonitor)]
        # adding new oracle monitors
        ora = Oracle(Config.CONNECTION_STRING)
        try:
            Row = namedtuple("Row", "name interval sql")
            sql = """
select
  name
, interval
, sql
from crm.monitor_sql
where enabled = 1
"""
            rows = ora.query(sql, lambda r: Row(r[0], r[1], r[2].read()))
            for row in rows:
                self.monitors[row.interval].append(OraMonitor(Config.CONNECTION_STRING, row.name, row.sql))
        except Exception:
            self.logger.exception("Error on initializing ora monitors")

        self.logger.debug("Ora initialization was completed")

    def load_es_monitors(self, interval):
        self.logger.debug("Loading ES monitors")

        ora = Oracle(Config.CONNECTION_STRING)
        try:
            Row = namedtuple("Row", "name index_name body_json exec_result")
            sql = """
select
  name
, index_name
, body_json
, exec_result
from crm.monitor_es
where enabled = 1
  and interval = :interval
"""
            rows = ora.query(sql, lambda r: Row(r[0], r[1], r[2].read(), r[3].read()), {"interval": interval})
            self.logger.debug("ES monitors were loaded")
            return rows
        except Exception:
            self.logger.exception("Error on initializing ora monitors")

    def dump_monitors(self):
        for key in self.monitors:
            self.logger.info("Monitors {0}:".format(key))
            for mon in self.monitors[key]:
                self.logger.info("\t{0}".format(mon))

    def exec_monitor(self, mon):
        try:
            sw = Stopwatch()
            val = mon.monitor()
            self.logger.info(f"Execute monitor. name: {mon.name}, value: {val}, elapsed: {sw.elapsed()}ms")
            gr = self.get_signal()
            gr.send(mon.name, val)
            return val
        except Exception:
            self.logger.exception(f"Error on executing monitor. name: {mon.name}, elapsed: {sw.elapsed()}ms")
            return []

    def run_monitors(self, interval_key):
        self.logger.info(f"Executing '{interval_key}' monitors. count: {len(self.monitors[interval_key])}")
        sw = Stopwatch()
        mon_results = {}
        for mon in self.monitors[interval_key]:
            mon_results[mon.name] = self.exec_monitor(mon)
        self.logger.info(f"Monitors '{interval_key}' were completed. count: {len(self.monitors[interval_key])}, elapsed: {sw.elapsed()}ms")
        self.add_mon_result(mon_results)
        return mon_results

    def add_mon_result(self, mon_dict):
        for key, val in mon_dict.items():
            self._monitor_results[key] = val

    def monitor_1min(self):
        self.logger.info("Runing 1min monitors.")
        sw = Stopwatch()
        partial_mons = self.run_monitors("1min")
        self.logger.info(f"SQL 1min monitors were completed. elapsed: {sw.elapsed()}ms")

        em = ElasticMonitorGroup(Config.ELASTICSEARCH_HOST, Config.ELASTICSEARCH_USER, Config.ELASTICSEARCH_PASSWORD,
                                 self.get_signal())
        elastic_mons = em.monitor_1min()
        self.add_mon_result(elastic_mons)
        external_mons = em.monitor_json(self.load_es_monitors("1min"))
        self.add_mon_result(external_mons)
        self.logger.info(f"Elastic 1min monitors were completed. elapsed: {sw.elapsed()}ms")

        return {**partial_mons, **elastic_mons, **external_mons}

    def monitor_5min(self):
        self.logger.info("Runing 5min monitors.")
        sw = Stopwatch()
        self.init_ora_mons()
        self.logger.info(f"init_ora_mons was completed. elapsed: {sw.elapsed()}ms")

        partial_mons = self.run_monitors("5min")
        self.logger.info(f"SQL 5min monitors were completed. elapsed: {sw.elapsed()}ms")

        em = ElasticMonitorGroup(Config.ELASTICSEARCH_HOST, Config.ELASTICSEARCH_USER, Config.ELASTICSEARCH_PASSWORD,
                                 self.get_signal())
        elastic_mons = em.monitor_5min()
        self.add_mon_result(elastic_mons)
        external_mons = em.monitor_json(self.load_es_monitors("5min"))
        self.add_mon_result(external_mons)
        self.logger.info(f"Elastic 5min monitors were completed. elapsed: {sw.elapsed()}ms")

        return {**partial_mons, **elastic_mons, **external_mons}

    def monitor_20min(self):
        self.logger.info("Runing 5min monitors.")
        sw = Stopwatch()
        result = self.run_monitors("20min")
        self.logger.info(f"SQL 20min monitors were completed. elapsed: {sw.elapsed()}ms")

        return result

    def monitor_1hour(self):
        self.logger.info("Runing 1hour monitors.")
        sw = Stopwatch()
        result = self.run_monitors("1hour")
        self.logger.info(f"SQL 1hour monitors were completed. elapsed: {sw.elapsed()}ms")

        return result

    def monitor_1day(self):
        self.logger.info("Runing 1day monitors.")
        sw = Stopwatch()
        result = self.run_monitors("1day")
        self.logger.info(f"SQL 1day monitors were completed. elapsed: {sw.elapsed()}ms")

        return result
