import logging
import requests
import os
import json


class TvmClient:

    def __init__(self):
        self.logger = logging.getLogger(self.__class__.__name__)
        self._tvm_token = TvmClient.__get_tvm_token()
        self._tvm_tool_url = TvmClient.__get_tvm_tool_url()
        self._ip = TvmClient.__get_host_ip()
        self.logger.debug(f"Host IP: {self._ip}")

    def get_service_ticket(self, app_name) -> str:
        auth_header = {"Authorization": self._tvm_token}
        response = requests.get(f"{self._tvm_tool_url}/tvm/tickets?dsts=" + app_name, headers=auth_header)
        try:
            data = json.loads(response.content)
            service_ticket = data[app_name]['ticket']
            return service_ticket
        except Exception:
            self.logger.exception(f"Error during parsing response. response: {response.content}")

    def get_user_ticket(self, oauth_token) -> str:
        user_ip = self._ip.split('/')[0]
        self.logger.debug(f"userip: {user_ip}")
        args = {"userip": user_ip}
        bb_url = "https://blackbox.yandex-team.ru/blackbox?method=oauth&get_user_ticket=yes&format=json"
        auth_header = {
            "X-Ya-Service-Ticket": self.get_service_ticket("blackbox"),
            "Authorization": "OAuth " + oauth_token
        }
        response = requests.get(bb_url, args, headers=auth_header)
        try:
            data = json.loads(response.content)
            user_ticket = data["user_ticket"]
            return user_ticket
        except Exception:
            self.logger.exception(f"Error during parsing response. response: {response.content}")

    @staticmethod
    def __get_tvm_token() -> str:
        token = os.environ.get("TVMTOOL_LOCAL_AUTHTOKEN")
        if token is not None:
            return token
        token = os.environ.get("QLOUD_TVM_TOKEN")
        if token is not None:
            return token
        raise RuntimeError("Unknown deployment system")

    @staticmethod
    def __get_tvm_tool_url():
        url = os.environ.get("DEPLOY_TVM_TOOL_URL")
        if url is not None:
            return url
        if os.environ.get("QLOUD_TVM_TOKEN") is None:
            raise RuntimeError("Unknown deployment system")
        return "http://localhost:1"

    @staticmethod
    def __get_host_ip():
        ip = os.environ.get("DEPLOY_POD_IP_0_ADDRESS")
        if ip is not None:
            return ip
        ip = os.environ("QLOUD_IPV6")
        if ip is not None:
            return ip
        raise RuntimeError("Unknown deployment system")
