package ru.yandex.crm.library.kotlin.service

import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.supervisorScope
import kotlinx.coroutines.withTimeout
import org.koin.core.Koin
import ru.yandex.crm.library.kotlin.service.CrmHostConstants.DEFAULT_SHUTDOWN_TIMEOUT_SECONDS
import java.util.concurrent.TimeUnit
import kotlin.time.Duration
import kotlin.time.ExperimentalTime

class CrmHost internal constructor(override val koin: Koin) : Host {
    companion object {
        fun createDefaultBuilder(args: Array<String>): HostBuilder = CrmHostBuilder(args)
            .withConfiguration()
            .withApplicationLifetime()
    }

    private val services: List<Service> = koin.getAll<Service>().distinctBy { it.javaClass }

    override suspend fun run() = coroutineScope {
        start()
        Runtime.getRuntime().addShutdownHook(ShutdownHook {
            koin.get<ApplicationLifetime>().stop()
        })
        koin.get<ApplicationLifetime>().start(launch { delay(Duration.INFINITE) })
        koin.get<ApplicationLifetime>().waitStop()
        stop()
    }

    override suspend fun start() = coroutineScope {
        for (service in services) {
            launch { service.start() }
        }
    }

    override suspend fun stop() {
        val exceptions = mutableListOf<Throwable>()
        val handler = CoroutineExceptionHandler { _, ex -> exceptions.add(ex) }
        supervisorScope {
            for (service in services) {
                launch(handler) {
                    withTimeout(TimeUnit.SECONDS.toMillis(DEFAULT_SHUTDOWN_TIMEOUT_SECONDS)) {
                        service.stop()
                    }
                }
            }
        }
        if (exceptions.isNotEmpty()) {
            val exception = Exception()
            for (ex in exceptions) {
                exception.addSuppressed(ex)
            }
            throw exception
        }
    }
}
