const { resolve, dirname, join } = require('path');
const stringifyObject = require('stringify-object');

function convertPluralKeys(keyset) {
  const result = {};

  for (const key of Object.keys(keyset || {})) {
    let value = keyset[key];

    if (Array.isArray(value)) {
      value = {
        one: value.length >= 1 ? value[0] : '',
        some: value.length >= 2 ? value[1] : '',
        many: value.length >= 3 ? value[2] : '',
        none: value.length >= 4 ? value[3] : '',
      };
    }

    result[key] = value;
  }

  return result;
}

const MODULES_SPLITTER = ':';
const COMPONENTS_SPLITTER = '~';
const COMMON_KEYSET = 'common';
function getPath(keyset, lang) {
  if (!keyset) {
    throw new Error(`Keyset ${keyset} isn't valid!`);
  }

  if (keyset === COMMON_KEYSET) {
    return resolve(`src/common.i18n/${lang}.ts`);
  }

  const pathname = keyset
    .split(MODULES_SPLITTER)
    .join('/modules/')
    .split(COMPONENTS_SPLITTER)
    .join('/components/');

  const filename = pathname.split('/').pop();

  return resolve(`src/${pathname}/${filename}.i18n/${lang}.ts`);
}

module.exports = function({ data }) {
  const langs = JSON.parse(data);
  const result = {};
  const indexes = {};

  for (const lang of Object.keys(langs)) {
    for (const keyset of Object.keys(langs[lang])) {
      const pathname = getPath(keyset, lang);
      const directory = dirname(pathname);
      result[pathname] = `export const ${lang} = ${stringifyObject(
        convertPluralKeys(langs[lang][keyset]),
        {
          indent: '  ',
          singleQuotes: true,
        },
      )};\n`;
      if (!indexes[directory]) {
        indexes[directory] = [];
      }
      indexes[directory].push(lang);
    }
  }

  for (const directory of Object.keys(indexes)) {
    result[join(directory, 'index.ts')] = indexes[directory]
      .map(lang => `export * from './${lang}';`)
      .join('\n');
  }

  return result;
};
