import os
import sys
import logging
import shutil
import subprocess
import boto3

from distutils.dir_util import copy_tree
from pathlib import Path

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG,
                    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')

aws_access_key_id = os.environ.get(
    'aws_access_key_id') if 'aws_access_key_id' in os.environ else ''
aws_secret_access_key = os.environ.get(
    'aws_secret_access_key') if 'aws_secret_access_key' in os.environ else ''

def _copyToStorage(workingDir: str, archiveName: str) -> None:
    session = boto3.session.Session()
    s3_client = session.client(
        service_name='s3',
        aws_access_key_id=aws_access_key_id,
        aws_secret_access_key=aws_secret_access_key,
        endpoint_url='http://s3.mds.yandex.net:80',
    )

    s3_client.upload_file(os.path.join(workingDir, archiveName),
                          "deploy", archiveName)


def _generateChangeLog(branch: str, workingDir: str) -> str:
    subprocess.check_call(
        ["git", "fetch", "--tags", "--force"], cwd=workingDir)
    master = subprocess.check_output(
        ["git", "describe", "--tags", "origin/master"], cwd=workingDir).decode("utf-8").strip()

    try:
        changeLog = subprocess.check_output(
            ["git", "log", "--pretty=\"%an: %s\"", f"{master}..{branch}"], cwd=workingDir).decode("utf-8").strip().replace("\"", "")
        
        logging.info(changeLog)

        return changeLog

    except:
        pass

    return ''


def main():
    args = sys.argv
    branch = args[1].split('/')[-1]
    workingDir = args[2]  # "%teamcity.build.workingDir%"
    deployPackageFolder = args[3]  # "%spaceDeployPackageLocataion%"
    buildNumber = args[4]  # "%build.number%"

    logging.info(f'workingDir {workingDir}')
    
    copy_tree(os.path.join(workingDir, "dist"),
                   os.path.join(deployPackageFolder, "dist"))
    copy_tree(os.path.join(workingDir, "vendor"),
                   os.path.join(deployPackageFolder, "vendor"))
    copy_tree(os.path.join(workingDir, "img"),
                   os.path.join(deployPackageFolder, "img"))
    shutil.copy(os.path.join(workingDir, "frontend.version"),
                deployPackageFolder)
    shutil.copy(os.path.join(workingDir, "index.html"),
                deployPackageFolder)
    
    changeLog = _generateChangeLog(branch, workingDir)
    with open(os.path.join(deployPackageFolder, "changeLog"), "w+") as f:
        f.write(changeLog)

    version = branch.split("-")[-1]
    archiveName = f'{branch}_front_{version}_{buildNumber}'
    shutil.make_archive(archiveName, 'zip', deployPackageFolder)
    archiveName += ".zip"
    logging.info(f"packaged to {archiveName}")
    _copyToStorage(workingDir, archiveName)
    logging.info("_copyToStorage finished")
    print(
        f"##teamcity[setParameter name='deployPackageLocation' value='{archiveName}']")


if __name__ == "__main__":
    main()
