import React, { FC } from 'react';
import cx from 'classnames';
import { Overlay } from 'components/Overlay';
import Menu from '@crm/components/dist/lego2/Menu';
import Dropdown from '@crm/components/dist/lego2/Dropdown';
import css from './ActionsPanel.module.css';
import { getActionsTree, actionToOption } from './ActionsPanel.utils';
import { ActionButton } from './ActionButton';
import { ActionsPanelProps } from './ActionsPanel.types';

export const ActionsPanel: FC<ActionsPanelProps> = ({
  direction = 'horizontal',
  onItemClick,
  actions,
  className,
  isLoading = false,
  view = 'pseudo',
  menuClassName,
  iconLeft,
}) => {
  if (!actions || actions.length === 0) {
    return null;
  }
  const actionsTree = getActionsTree(actions);
  const handleMenuItemChange = ({ target }) => {
    const { value } = target;
    const action = actions.find(({ id }) => id === value);
    if (action) {
      onItemClick?.(action);
    }
  };
  const isVertical = direction === 'vertical';
  const panelClassNames = cx(
    css.ActionsPanel,
    { [css.ActionsPanel__verticalDirection]: isVertical },
    className,
  );

  return (
    <div data-testid="ActionsPanel" className={panelClassNames}>
      <Overlay display={isLoading} color="rgba(255,255,255,.7)" zIndex={10} />
      {[...actionsTree.keys()].map((actionAsKey) => {
        const actionChildren = actionsTree.get(actionAsKey);
        if (actionChildren !== undefined) {
          return (
            <Dropdown
              key={actionAsKey.id}
              view={view}
              iconLeft={iconLeft}
              hasArrow
              popupDirections={['bottom-right']}
              content={
                <Menu
                  className={menuClassName}
                  items={actionChildren.map(actionToOption)}
                  onChange={handleMenuItemChange}
                />
              }
            >
              {actionAsKey.caption}
            </Dropdown>
          );
        }
        return (
          <ActionButton
            key={actionAsKey.id}
            onClick={onItemClick}
            theme="clear"
            action={actionAsKey}
            width={isVertical ? 'max' : 'auto'}
          />
        );
      })}
    </div>
  );
};
