import React, { useContext, useEffect, useState, FC, HTMLAttributes } from 'react';
import { timer, throwError } from 'rxjs';
import { share, switchMapTo } from 'rxjs/operators';
import { AsyncRenderContext } from './AsyncRender.context';
import { AsyncRender } from './AsyncRender';

interface SimpleAsyncComponentProps extends HTMLAttributes<HTMLDivElement> {
  delay?: number;
  isError?: boolean;
}

const useAsyncTask = (delay = 2000, isError = false) => {
  const asyncRender = useContext(AsyncRenderContext);
  const [isCompleted, setCompleted] = useState(false);

  useEffect(() => {
    if (isCompleted) {
      return;
    }

    let task = timer(delay);
    if (isError) {
      task = task.pipe(switchMapTo(throwError(new Error('error'))));
    }
    task = task.pipe(share());

    const subscription = task.subscribe({
      complete: () => {
        setCompleted(true);
      },
      error: () => {},
    });
    asyncRender.registerTask(task);

    return () => {
      subscription.unsubscribe();
      asyncRender.unregisterTask(task);
    };
  }, [asyncRender]);

  return isCompleted;
};

export const SimpleAsyncComponent: FC<SimpleAsyncComponentProps> = ({
  title = 'SimpleAsyncComponent',
  delay = 2000,
  isError = false,
  children,
  ...props
}) => {
  const isComplete = useAsyncTask(delay, isError);

  return (
    <div {...props}>
      <div>{title}</div>
      {isComplete && <div>{children}</div>}
    </div>
  );
};

export const Nested = (_args) => {
  return (
    <div style={{ position: 'relative', height: 200, width: 300 }}>
      <AsyncRender>
        <div style={{ position: 'relative', height: 100, width: 300 }}>
          <SimpleAsyncComponent title="Timeline" />
        </div>
        <div style={{ position: 'relative', height: 100, width: 300 }}>
          <AsyncRender totalSubTasks={2}>
            <SimpleAsyncComponent title="MailForm">
              <SimpleAsyncComponent title="MailEditor" />
            </SimpleAsyncComponent>
          </AsyncRender>
        </div>
      </AsyncRender>
    </div>
  );
};

export const WithErrorChildrenRender = (_args) => {
  return (
    <AsyncRender>
      <SimpleAsyncComponent title="CommentForm" isError />
    </AsyncRender>
  );
};

export const WithSyncChildrenRender = (_args) => {
  return (
    <AsyncRender totalSubTasks={0}>
      <div>MessageForm</div>
    </AsyncRender>
  );
};

export const WithForceShowSpinner = (_args) => {
  return (
    <div style={{ position: 'relative', height: 100, width: 300 }}>
      <AsyncRender totalSubTasks={0} forceShowSpinner>
        <div>Content for transition</div>
      </AsyncRender>
    </div>
  );
};
