import React, { useEffect, useState, useCallback } from 'react';
import { BatchTask } from 'services/BatchTask';
import { useObservable } from 'components/useObservable';
import { Hidden } from 'components/Hidden';
import { RetryButton } from 'components/RetryButton';
import Spinner from 'components/Spinner';
import css from './AsyncRender.module.css';
import { AsyncRenderContext } from './AsyncRender.context';
import { testIds } from './AsyncRender.config';
import { AsyncRenderProps } from './AsyncRender.types';

export const AsyncRender = ({
  children,
  className,
  style,
  totalSubTasks,
  retryComponent: RetryComponent = RetryButton,
  onRenderSuccess,
  onRenderError,
  hidden = false,
  forceShowSpinner,
}: AsyncRenderProps) => {
  const [key, setKey] = useState(1);
  const [batchTask, setBatchTask] = useState(() => new BatchTask({ totalSubTasks }));
  const [wasCompleted, setWasCompleted] = useState(totalSubTasks === 0);
  const { error, complete: isCompleted } = useObservable(batchTask.asObservable);

  useEffect(() => {
    if (isCompleted) {
      setWasCompleted(true);
      onRenderSuccess?.();
    }
  }, [isCompleted]);

  useEffect(() => {
    if (error) {
      onRenderError?.(error);
    }
  }, [error]);

  useEffect(() => {
    return () => {
      batchTask.destroy();
    };
  }, [batchTask]);

  const handleRetry = useCallback(() => {
    setKey(key + 1);
    setBatchTask(new BatchTask({ totalSubTasks }));
  }, [key, totalSubTasks]);

  const isHidden = Boolean(!wasCompleted || error);
  const isShowError = !hidden && Boolean(error);
  const isShowSpinner = !hidden && (forceShowSpinner || (!error && !isCompleted));

  return (
    <AsyncRenderContext.Provider value={batchTask}>
      <Hidden
        data-testid={testIds.hiddenAllWrap}
        className={className}
        style={style}
        hidden={hidden}
      >
        <Hidden data-testid={testIds.hiddenChildrenWrap} hidden={isHidden} key={key}>
          {children}
        </Hidden>
        {isShowError && (
          <div data-testid={testIds.errorBlock} className={css.AsyncRenderContainer__errorBlock}>
            <RetryComponent onRetry={handleRetry} />
          </div>
        )}
        {isShowSpinner && <Spinner modal={false} overlay visible delay asComponent />}
      </Hidden>
    </AsyncRenderContext.Provider>
  );
};
