import React, { useEffect, useCallback, useMemo, useState, useRef } from 'react';
import { AsyncRender } from 'components/AsyncRender';
import { AsyncTransitionProps, AsyncTransitionState } from './AsyncTransition.types';

export const AsyncTransition = <S extends AsyncTransitionState>({
  onTransitionError,
  onTransitionSuccess,
  children,
  state,
  className,
  style,
}: AsyncTransitionProps<S>) => {
  const memoState = useMemo(() => state, [state.hash]);
  const prevState = useRef(memoState);
  const [currentState, setCurrentState] = useState(memoState);
  const [nextState, setNextState] = useState(memoState);
  const [wasCompleted, setWasCompleted] = useState(false);

  const handleNextStateComplete = useCallback(() => {
    prevState.current = currentState;
    setCurrentState(memoState);
  }, [memoState]);
  const handleNextStateError = useCallback(() => {
    onTransitionError?.(currentState);
    setNextState(currentState);
  }, [currentState, onTransitionError]);

  useEffect(() => {
    setNextState(memoState);

    if (!wasCompleted || memoState.isForceMountNextState) {
      handleNextStateComplete();
    }
  }, [memoState]);

  useEffect(() => {
    onTransitionSuccess?.({ currentState, prevState: prevState.current });
  }, [currentState]);

  const handleCurrentStateComplete = useCallback(() => {
    setWasCompleted(true);
  }, []);

  return (
    <>
      <AsyncRender
        key={currentState.hash}
        className={className}
        style={style}
        forceShowSpinner={currentState !== nextState}
        onRenderSuccess={handleCurrentStateComplete}
        totalSubTasks={currentState.totalSubTasks}
      >
        {children(currentState)}
      </AsyncRender>
      {nextState !== currentState && (
        <AsyncRender
          key={nextState.hash}
          hidden
          onRenderSuccess={handleNextStateComplete}
          onRenderError={handleNextStateError}
          totalSubTasks={nextState.totalSubTasks}
        >
          {children(nextState)}
        </AsyncRender>
      )}
    </>
  );
};
