import Bluebird from 'bluebird';
import { FormApi } from 'final-form';
import * as api from 'lego/components/AttachFiles/api';
import { config } from 'services/Config';
import { BackendResponse } from 'lego/components/AttachFiles';
import { makeAutoObservable } from 'mobx';
import { AttachFilesServiceProps } from './AttachFilesService.types';

export class AttachFilesService implements AttachFilesServiceProps {
  public isFetch: boolean = false;

  public name: string;
  public objectName: string;
  public containerId?: number;
  public objId?: number;
  public formApi?: FormApi;
  public promise: Bluebird<void> | null;

  constructor() {
    makeAutoObservable(this);
  }

  public init = (
    name: string,
    objectName: string,
    formApi?: FormApi,
    containerId?: number,
    objId?: number,
  ) => {
    this.name = name;
    this.objectName = objectName;
    this.formApi = formApi;
    this.containerId = containerId;
    this.objId = objId;
  };

  public addFiles = (files: File[]) => {
    if (files.length === 0) {
      return null;
    }

    this.cancelPromise();
    this.setFetch(true);

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const args: any = {
      files,
      objectName: this.objectName,
      containerId: this.containerId,
    };

    if (!config.value.features.newEditingFiles || !['internalchatmessage', 'chatmessage'].includes(this.objectName)) {
      args.objId = this.objId;
    }

    this.promise = api
      .add(args)
      .then(this.handleFilesChange)
      .finally(() => {
        this.setFetch(false);
      });

    return this.promise;
  };

  public updateFiles = (fileIdsToLink: number[], fileIdsToRemove?: number[]) => {
    this.cancelPromise();
    this.setFetch(true);

    this.promise = api
      .update({
        fileIdsToLink,
        fileIdsToRemove,
        objectName: this.objectName,
        containerId: this.containerId,
        objId: this.objId,
      })
      .then(this.handleFilesChange)
      .finally(() => {
        this.setFetch(false);
      });

    return this.promise;
  };

  public removeFile = (id) => {
    this.cancelPromise();
    this.setFetch(true);

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const apiArgs: any = {
      fileId: id,
      containerId: this.containerId,
    };
    let apiRequest = api.removeFileFromContainer;

    if (!config.value.features.newEditingFiles || !['internalchatmessage', 'chatmessage'].includes(this.objectName)) {
      apiRequest = api.remove;
      apiArgs.objectName = this.objectName;
      apiArgs.objId = this.objId;
    }

    this.promise = apiRequest(apiArgs)
      .then(this.handleFilesChange)
      .finally(() => {
        this.setFetch(false);
      });

    return this.promise;
  };

  private cancelPromise = () => {
    if (this.promise) {
      this.promise.cancel();
    }
  };

  private setFetch = (isFetch) => {
    this.isFetch = isFetch;
    this.formApi?.mutators.setLoading?.(isFetch);
  };

  public handleFilesChange = (data: BackendResponse) => {
    if (!data.files) {
      return;
    }

    this.formApi?.batch(() => {
      this.formApi?.change(this.name, data);
      this.formApi?.change('containerId', data.containerId);
    });
  };
}
