import { AttachmentItem } from 'components/AttachmentItem';
import React, { FC, MouseEvent, useRef } from 'react';
import { useEffect } from 'react';
import cx from 'classnames';
import css from './AttachmentListStateless.module.css';
import { AttachmentListStatelessProps } from './AttachmentList.types';

const FLEX_GAP_VALUE = 8;

export const AttachmentListStateless: FC<AttachmentListStatelessProps> = (props) => {
  const {
    className,
    files,
    onFileClick,
    onFileDelete,
    isOpen,
    onHiddenItemsChange,
    hiddenFilesCount,
  } = props;

  const ref = useRef<HTMLDivElement>(null);

  const calculateItemsCount = (parent: Element) => {
    const children = Array.from(parent.children);
    const parentWidth = parent.clientWidth;
    let childrenWidth = 0;

    for (let currentChild of children) {
      childrenWidth += currentChild.clientWidth + FLEX_GAP_VALUE;
      if (childrenWidth - FLEX_GAP_VALUE > parentWidth) {
        return children.indexOf(currentChild);
      }
    }

    return children.length;
  };

  const resizeObserver = new ResizeObserver((entries) => {
    if (onHiddenItemsChange) {
      const count = calculateItemsCount(entries[0].target);

      if (hiddenFilesCount !== files.length - count) {
        if (count >= files.length) {
          onHiddenItemsChange(0);
        } else {
          onHiddenItemsChange(files.length - count);
        }
      }
    }
  });

  useEffect(() => {
    resizeObserver.observe(ref.current as Element);
    return () => resizeObserver.disconnect();
  });

  const classNames = cx(css.AttachmentListStateless, className, {
    [css.AttachmentListStateless_closed]: !isOpen,
  });

  return (
    <div
      className={classNames}
      style={{
        gap: `${FLEX_GAP_VALUE}px`,
      }}
      ref={ref}
    >
      {Array.isArray(files) &&
        Boolean(files.length) &&
        files.map((file) => (
          <AttachmentItem
            {...file}
            key={file.id}
            fileSize={file.size}
            onClick={onFileClick ? (event: MouseEvent) => onFileClick(event, file) : undefined}
            onDelete={onFileDelete ? (event: MouseEvent) => onFileDelete(event, file) : undefined}
          />
        ))}
    </div>
  );
};
