import * as React from 'react';
import { WithOId } from 'types';
import cx from 'classnames';
import isEmptyValue from 'utils/isEmptyValue';
import Access from 'utils/Access';
import Adapter from 'utils/Adapter';
import { AnyObject } from 'final-form';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import Attribute from './Attribute';
import Popup from './Popup';
import css from './AttributeWithPopup.scss';
import { Value, AttributeMeta } from './types';
import { attributesAdapter } from './adapters';
import { IdNameRecord } from './UxExperimentSplitter/UxExperimentSplitter.types';

export interface Props extends Partial<WithOId> {
  className?: string;
  label?: string;
  value?: number;
  access?: number;
  theme?: string;
  getButtonRef?: React.Ref<HTMLButtonElement>;
  isNewDesign?: boolean;
  isPopup?: boolean;
  onLabelClick?: () => void;
  onCancel?: () => void;
  onChange?: (value: Value) => unknown;
  onClear?: () => unknown;
  onAdd?: (value: Value) => unknown;
  onRemove?: (value: Value) => unknown;
  onItemUpdate?: (value: Value) => unknown;
  component: string;
  size?: LegoSizeProp;
  fieldProps?: AnyObject;
  adapter?: Adapter<AttributeMeta>;
  updateSubscriber?: (value: IdNameRecord | null) => void;
}

const AttributeWithPopup = (props: Props): React.ReactElement => {
  const {
    className,
    label,
    value,
    theme,
    getButtonRef,
    isPopup,
    onLabelClick,
    onCancel,
    onChange,
    onAdd,
    onClear,
    onRemove,
    onItemUpdate,
    component,
    access,
    size,
    fieldProps,
    adapter = attributesAdapter,
    oId,
    updateSubscriber,
  } = props;

  const isBlock = theme === 'column' && !isEmptyValue(value);

  const FastActionComponent = attributesAdapter.getOrDefault(component).fastActionComponent;

  return (
    <div
      data-testid={`attribute_${component}`}
      className={cx(className, { [css.b_block]: isBlock })}
    >
      <Attribute
        label={label}
        value={value}
        theme={theme}
        getButtonRef={getButtonRef}
        hideValue={isPopup}
        onLabelClick={onLabelClick}
        access={access}
        component={component}
        fieldProps={fieldProps}
        adapter={adapter}
        addonUnderValue={
          FastActionComponent &&
          Access.isEdit(access) && <FastActionComponent value={value} onChange={onChange} />
        }
      />
      {isPopup && (
        <Popup
          onCancel={onCancel}
          onChange={onChange}
          onAdd={onAdd}
          onClear={onClear}
          onRemove={onRemove}
          onItemUpdate={onItemUpdate}
          value={value}
          component={component}
          size={size}
          fieldProps={fieldProps}
          oId={oId}
          updateSubscriber={updateSubscriber}
          adapter={adapter}
        />
      )}
    </div>
  );
};

AttributeWithPopup.defaultProps = {
  theme: 'row',
};

export default AttributeWithPopup;
