import React, { Component, ComponentClass } from 'react';
import UXExperiment from 'services/Rum/UXExperiment';
import AttributeWithPopup, { Props } from '../AttributeWithPopup';
import { UxExperimentSplitterProps } from './UxExperimentSplitter.types';

import {
  participants,
  getExperimentConfig,
  getUsefulPayload,
  userFlowCheckPoints,
} from './UxExperimentSplitter.config';

class UxExperimentSplitter extends Component<UxExperimentSplitterProps> {
  private experiment: UXExperiment;
  constructor(props) {
    super(props);
    const attributeName = props.name;
    if (this.inExperiment(attributeName)) {
      this.experiment = new UXExperiment(getExperimentConfig(attributeName));
    }
  }

  private getUsefulPayload = () => {
    return getUsefulPayload(this.props.currentIssueInfo.data);
  };

  private addCheckPoint = (name: string, payload = {}): void => {
    this.experiment.checkPoint(name, Date.now(), { ...this.getUsefulPayload(), ...payload });
  };

  private addCheckpoints = () => {
    if (this.experiment.isStarted) {
      this.addCheckPoint(userFlowCheckPoints.cancel);
    } else {
      this.addCheckPoint(userFlowCheckPoints.open);
    }
  };

  private toggleEditing = () => {
    if (this.props.onLabelClick) {
      this.props.onLabelClick();
    }
  };

  private onChange = (value) => {
    const { onChange, name } = this.props;
    if (this.experiment.isStarted) {
      const componentValue = { [name]: value };
      this.addCheckPoint(userFlowCheckPoints.save, { componentValue });
    }
    if (typeof onChange === 'function') {
      return onChange(value);
    }
  };

  private onCancel = () => {
    const { onCancel } = this.props;
    if (this.experiment.isStarted) {
      this.addCheckPoint(userFlowCheckPoints.cancel);
    }
    if (typeof onCancel === 'function') {
      onCancel();
    }
  };

  // Костыль чтобы подписаться на события очистки саджеста и смены значения
  private suggestUpdateSubscriber = (value): void => {
    if (this.experiment.isStarted) {
      if (value === null) {
        this.addCheckPoint(userFlowCheckPoints.clearValue);
        return;
      }
      this.addCheckPoint(userFlowCheckPoints.setValue);
    }
  };

  private inExperiment = (name: string) => participants.includes(name);

  private getUxProps = () => {
    const { name } = this.props;
    if (this.inExperiment(name)) {
      return {
        toggleEditing: this.toggleEditing,
        addCheckpoints: this.addCheckpoints,
        onChange: this.onChange,
        onCancel: this.onCancel,
        updateSubscriber: this.suggestUpdateSubscriber,
        handlersWithoutTimeMark: {
          onLabelClick: this.props.onLabelClick,
          onChange: this.props.onChange,
          onCancel: this.props.onCancel,
        },
      };
    }
    return {};
  };

  render() {
    const UxProps = this.getUxProps();
    return <AttributeWithPopup {...this.props} {...UxProps} />;
  }
}

export default UxExperimentSplitter as ComponentClass<Props>;
