import React, { FC, useState, useEffect, useRef, useCallback } from 'react';
import { useOutsideClick } from '@crm/components/dist/useOutsideClick';
import { isBefore } from 'date-fns';
import { useIMask } from '@crm/react-imask';
import { Instance } from '@popperjs/core';
import { DatePicker } from 'components/DatePicker';
import { useKeyboard } from '@crm/components/dist/useKeyboard';
import { BubbleInput } from '@crm/components/dist/BubbleInput';
import { Popup } from '@crm/components/dist/Popup';
import { EditingArea } from '@crm/components/dist/Attribute2/components/EditingArea';
import { DateEditingProps, DateEditingValue } from '../Date.types';
import css from './Editing.module.css';
import { parseMaskValueToDate, areTimestampsDifferent } from './Editing.utils';
import { DEFAULT_MIN_DATE } from './Editing.config';

export const Editing: FC<DateEditingProps> = (props) => {
  const {
    tabIndex,
    minDate,
    editingRef,
    display = true,
    label,
    name,
    editingValue,
    onEditingChange,
    onChange,
  } = props;

  const [selectedDate, setSelectedDate] = useState<Date | null>(
    editingValue?.length ? new Date(editingValue) : null,
  );
  const popperRef = useRef<Instance>();
  const [text, setText] = useState(() => {
    if (editingValue) {
      return new Date(editingValue).toLocaleDateString('ru');
    }
    return '';
  });
  const editingAreaRef = useRef<HTMLDivElement | null>();
  const contentEditableRef = useRef<HTMLDivElement>();
  const { ref, maskRef } = useIMask({
    mask: Date,
  });

  const emitChange = useCallback(
    (newValue: DateEditingValue) => {
      const isInvertingValue = Boolean((!editingValue && newValue) || (editingValue && !newValue));
      const valuesDiffer =
        isInvertingValue ||
        Boolean(editingValue && newValue && areTimestampsDifferent(editingValue, newValue));
      if (valuesDiffer) {
        onChange?.(newValue);
      }
      onEditingChange?.(false);
    },
    [onChange, editingValue, onEditingChange],
  );

  useOutsideClick(
    editingAreaRef,
    useCallback(() => {
      if (!display) {
        return;
      }

      const mask = maskRef.current;
      if (!mask.value) {
        emitChange?.(null);
      } else {
        emitChange(parseMaskValueToDate(mask.value).toISOString());
      }
    }, [display, emitChange]),
    false,
    'click',
  );

  useKeyboard(
    {
      onEsc: () => onEditingChange?.(false),
      onTab: (event: KeyboardEvent) => event.preventDefault(),
      onEnterCapture: () => {
        if (!text.length) {
          emitChange?.(null);
        } else {
          emitChange?.(selectedDate && selectedDate.toISOString());
        }
      },
    },
    display,
    [onEditingChange, onChange, text],
  );

  useEffect(() => {
    const mask = maskRef.current;
    const handleMaskComplete = () => {
      const parsedDate = parseMaskValueToDate(mask.value);
      if (minDate && isBefore(parsedDate, new Date(minDate))) {
        setSelectedDate(new Date(minDate));
        return;
      }
      setSelectedDate(parsedDate);
    };

    mask?.on('complete', handleMaskComplete);
    return () => {
      mask?.off('complete', handleMaskComplete);
    };
  }, []);

  useEffect(() => {
    if (display && editingValue) {
      setText(new Date(editingValue).toLocaleDateString('ru'));
      setSelectedDate(new Date(editingValue));

      return () => {
        setText('');
        setSelectedDate(null);
      };
    }
  }, [display]);

  const handleContentEditableRef = useCallback(
    (instance: HTMLDivElement) => {
      editingRef?.(instance);
      contentEditableRef.current = instance;
      ref.current = instance as HTMLInputElement;
    },
    [editingRef],
  );

  const handleTextinputChange = (value: string) => {
    popperRef.current?.update();
    setText(value);
  };

  const handleDatePickerSelect = (date: Date | null) => {
    if (!date) {
      return;
    }

    setText(date.toLocaleDateString('ru'));
    setSelectedDate(date);
    emitChange?.(date.toISOString());
  };

  return (
    <EditingArea
      name={name}
      display={display}
      tabIndex={-1}
      innerRef={(instance) => (editingAreaRef.current = instance)}
    >
      <BubbleInput
        display={display}
        text={text}
        tabIndex={tabIndex}
        placeholder={label}
        onTextChange={handleTextinputChange}
        contentEditableRef={handleContentEditableRef}
        autoFocus
        hasClear
      />
      <Popup
        isVisible={display}
        anchor={editingAreaRef.current}
        direction="bottom-start"
        className={css.Editing__popup}
        sameWidthProperty="max-width"
      >
        <DatePicker
          key={String(display)} // сбрасываем к первоначальному состоянию при переоткрытии
          view="mg"
          selected={selectedDate}
          minDate={minDate ? new Date(minDate) : DEFAULT_MIN_DATE} // с null дейтпикер почему-то не показывает выбранную дату. https://st.yandex-team.ru/CRM-17761
          onSelect={handleDatePickerSelect}
          disabledKeyboardNavigation
          fixedHeight
        />
      </Popup>
    </EditingArea>
  );
};
