import React, { FC, useCallback, useState, useRef, useEffect } from 'react';
import { BubbleProps } from '@crm/components/dist/Bubble';
import { Instance } from '@popperjs/core';
import { PopupSkills, LevelType } from '../../components/PopupSkills';
import {
  Editing as SuggestSelectMultipleEditing,
  RenderPopupMenuProps,
  SuggestSelectMultipleEditingInstance,
} from '@crm/components/dist/Attribute2/SuggestSelectMultiple';
import { SkillsEditingProps } from '../Skills.types';
import { SkillBubble } from '../SkillBubble';
import { ValuedSkill } from '../../types/ValuedSkill';
import { Skill } from '../../types/Skill';
import { useInsideAttributesKeyboard } from '@crm/components/dist/Attribute2/components/InsideAttributesKeyboardContext';

export const Editing: FC<SkillsEditingProps> = (props) => {
  const { display, onChange, editingValue, editingRef } = props;

  const contentEditableRef = useRef<HTMLDivElement>();
  const popperRef = useRef<Instance>();
  const instanceRef = useRef<SuggestSelectMultipleEditingInstance>();
  const [popupSkillsCurrentSkill, setPopupSkillsCurrentSkill] = useState<Skill>();
  const [popupSkillsLevel, setPopupSkillsLevel] = useState<LevelType>();

  const insideKeyboard = useInsideAttributesKeyboard();
  useEffect(() => {
    if (display) {
      const finishHandling = insideKeyboard.startHandling();
      return finishHandling;
    }
  }, [display]);

  const handleBubbleClick = (skill: ValuedSkill) => {
    setPopupSkillsCurrentSkill(skill);
    setPopupSkillsLevel('value');
  };

  const handleLevelChange = (level: 'skill' | 'value') => {
    setPopupSkillsLevel(level);
  };

  const renderSelectedBubble = (skill: ValuedSkill, props: BubbleProps) => {
    return (
      <SkillBubble skill={skill} onClick={() => handleBubbleClick(skill)} canDelete {...props} />
    );
  };

  const handleEditingRef = useCallback(
    (instance: HTMLDivElement) => {
      contentEditableRef.current = instance;
      editingRef?.(instance);
    },
    [editingRef],
  );

  const handlePopupSkillsChange = (skill: ValuedSkill) => {
    setTimeout(() => contentEditableRef.current?.focus(), 0);
    instanceRef.current?.resetText();
    instanceRef.current?.reload();
    const index = editingValue.findIndex((s) => s.id === skill.id);
    const hasSkillAlready = index !== -1;
    if (hasSkillAlready) {
      const updatedClone = editingValue.slice();
      updatedClone.splice(index, 1, skill);
      onChange?.(updatedClone);
      return;
    }

    onChange?.([...editingValue, skill]);
  };

  const renderPopupMenu = useCallback(
    (props: RenderPopupMenuProps<Skill>) => {
      return (
        <PopupSkills
          {...props}
          level={popupSkillsLevel}
          onLevelChange={handleLevelChange}
          currentSkill={popupSkillsCurrentSkill}
          onCurrentSkillChange={setPopupSkillsCurrentSkill}
          items={props.items?.filter((item) => !editingValue.find((ev) => ev.id === item.id))}
          onChange={handlePopupSkillsChange}
        />
      );
    },
    [editingValue, popupSkillsLevel, popupSkillsCurrentSkill],
  );

  return (
    <SuggestSelectMultipleEditing<Skill>
      {...props}
      canListenKeyboard={popupSkillsLevel !== 'value'}
      instanceRef={(instance) => (instanceRef.current = instance)}
      isContentEditableDisabled={popupSkillsLevel === 'value'}
      renderPopupMenu={renderPopupMenu}
      editingRef={handleEditingRef}
      renderSelectedBubble={renderSelectedBubble}
      onChange={onChange}
      popperRef={(instance) => (popperRef.current = instance)}
    />
  );
};
