import React, { FC, useCallback, useState, useRef, useEffect, MutableRefObject } from 'react';
import NewTagModal from 'modules/tags/components/NewTagModal';
import { BubbleProps } from '@crm/components/dist/Bubble';
import { tagCreatedEvent } from 'utils/events/tagCreatedEvent';
import { Instance } from '@popperjs/core';
import { TagBubble } from 'components/TagBubble';
import connectNewTagModal from 'modules/tags/containers/connectNewTagModal';
import {
  Editing as SuggestSelectMultipleEditing,
  RenderPopupMenuProps,
  SuggestSelectMultipleEditingInstance,
} from '@crm/components/dist/Attribute2/SuggestSelectMultiple';
import { PopupTagCollection } from '../../components/PopupTagCollection';
import { TagsEditingProps, Tag } from '../Tags.types';

const renderSelectedTagBubble = (tag: Tag, props: BubbleProps) => {
  return <TagBubble tag={tag} {...props} canDelete />;
};
const TagModal = connectNewTagModal(NewTagModal);

export const Editing: FC<TagsEditingProps> = (props) => {
  const { onChange, editingValue, editingRef: outerEditingRef, display, onEvent } = props;
  const [initialValues, setInitialValues] = useState<Record<string, string>>({});
  const popperRef = useRef<Instance>();
  const instanceRef = useRef<SuggestSelectMultipleEditingInstance>();
  const editingRef = useRef<HTMLElement>();
  const [isModalVisible, setModalVisible] = useState(false);

  useEffect(() => {
    if (!display) {
      setModalVisible(false);
      setInitialValues({});
    }
  }, [display]);

  const handleCreate = (text: string) => {
    setInitialValues({
      name: text,
    });
    setModalVisible(true);
  };

  const handleHide = () => {
    setInitialValues({});
    setModalVisible(false);
  };

  useEffect(() => {
    popperRef.current?.update();
  }, [editingValue.length]);

  const handleTagAdd = (tagDTO: { id: number; name: string; color?: string }) => {
    const newTag: Tag = {
      id: tagDTO.id,
      text: tagDTO.name,
      color: tagDTO.color && `#${tagDTO.color}`,
      type: 'Personal',
    };
    instanceRef.current?.reload();
    instanceRef.current?.resetText();
    const newTags = [...editingValue, newTag];
    onChange?.(newTags);
    onEvent?.(tagCreatedEvent());
    editingRef.current?.focus();
  };

  const handleEditingRef = useCallback(
    (instance: HTMLElement) => {
      editingRef.current = instance;
      outerEditingRef?.(instance);
    },
    [outerEditingRef],
  );

  const renderPopupCollection = useCallback(
    ({ items, text, ...props }: RenderPopupMenuProps<Tag>) => (
      <PopupTagCollection
        {...props}
        scope={(outerEditingRef as unknown) as MutableRefObject<HTMLElement>}
        common={items.filter(
          (tag) => tag.type === 'Shared' && !editingValue.find((item) => item.id === tag.id),
        )}
        personal={items.filter(
          (tag) => tag.type === 'Personal' && !editingValue.find((item) => item.id === tag.id),
        )}
        creationText={text.trim()}
        onCreate={handleCreate}
      />
    ),
    [editingValue, outerEditingRef],
  );

  const handlePopperRef = useCallback((instance: Instance) => {
    popperRef.current = instance;
  }, []);

  return (
    <SuggestSelectMultipleEditing
      {...props}
      popperRef={handlePopperRef}
      instanceRef={(instance) => (instanceRef.current = instance)}
      canListenKeyboard={!isModalVisible}
      canFocusMenu={!isModalVisible}
      editingRef={handleEditingRef}
      renderSelectedBubble={renderSelectedTagBubble}
      renderPopupMenu={renderPopupCollection}
      appendAddon={
        <TagModal
          form="NewTag"
          onHide={handleHide}
          initialValues={initialValues}
          addTag={handleTagAdd}
          show={isModalVisible}
        />
      }
    />
  );
};
