import React, { FC, useState, useCallback } from 'react';
import Form from 'modules/timer/components/Form';
import Modal from '@crm/components/dist/lego2/Modal';
import { stopEditingEvent, startEditingEvent, changeEvent } from '@crm/components/dist/events';
import { Reading } from './Reading';
import { Editing } from './Editing';
import { TimerProps, TimerSubmitValues } from './Timer.types';
import { EntireAttribute } from '@crm/components/dist/Attribute2/components/EntireAttribute';

export const Timer: FC<TimerProps> = (props) => {
  const { editingValue, onSubmit, onEditingChange, onEvent, isEditing } = props;

  const [isModalVisible, setModalVisible] = useState(false);

  const handleEditingChange = useCallback(
    (isEditing: boolean) => {
      if (!editingValue && isEditing) {
        setModalVisible(true);
        return;
      }

      onEditingChange?.(isEditing);
    },
    [onEditingChange, editingValue],
  );

  const handleModalClose = useCallback(() => {
    if (!isEditing) {
      onEvent?.(stopEditingEvent());
    }
    setModalVisible(false);
  }, [isEditing]);

  const handleExistingTimerClick = useCallback(() => {
    if (!isEditing) {
      onEvent?.(startEditingEvent());
    }
    setModalVisible(true);
  }, [isEditing]);

  const handleSubmit = useCallback(
    (values: TimerSubmitValues) => {
      onEvent?.(changeEvent());
      onSubmit?.(values);
      setModalVisible(false);
    },
    [onSubmit],
  );

  return (
    <>
      <EntireAttribute
        {...props}
        onEditingChange={handleEditingChange}
        onExistingTimerClick={handleExistingTimerClick}
        isModalVisible={isModalVisible}
        reading={Reading}
        editing={Editing}
      />
      <Modal visible={isModalVisible} onClose={handleModalClose}>
        <Form
          // @ts-ignore
          form="timer"
          formType="issues"
          module="cases"
          isNew={!editingValue}
          initialValues={{ ...editingValue, startDate: editingValue?.activationDate }}
          onCancel={handleModalClose}
          onSubmit={handleSubmit}
        />
      </Modal>
    </>
  );
};
