import React, { useState, useCallback, FC } from 'react';
import { stopEditingEvent } from '@crm/components/dist/events';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';
import { Event } from '@crm/components/dist/types/Event';
import {
  Account,
  AccountSubmitValues,
  AccountEditingValue,
  AccountFeatureFlags,
} from 'components/Attribute2';
import { EObject } from 'types/entities';

export const AccountInput: FC<AttributeComponentProps & {
  EObject?: EObject;
  featureFlags?: AccountFeatureFlags;
}> = (props) => {
  const { attribute, onChange, isDisabled, onEvent, EObject, featureFlags } = props;
  const value = (props.value as AccountEditingValue) || null;
  const [isReadingLoading, setReadingLoading] = useState(false);
  const [isEditing, setEditing] = useState(false);

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const changeValue = useCallback(
    async (accountId: number | string | null) => {
      setReadingLoading(true);
      if (isEditing) {
        setEditing(false);
        emitEvent(stopEditingEvent());
      }
      try {
        await onChange(attribute, accountId);
      } finally {
        setReadingLoading(false);
      }
    },
    [onChange, emitEvent, isEditing],
  );

  const handleSubmit = useCallback(
    (values: AccountSubmitValues) => {
      return changeValue(values.id);
    },
    [changeValue],
  );

  const handleChange = useCallback(
    (value: AccountEditingValue) => {
      return changeValue(value && value.id);
    },
    [changeValue],
  );

  return (
    <Account
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      isReadingLoading={isReadingLoading}
      readingValue={value}
      editingValue={value}
      onSubmit={handleSubmit}
      onChange={handleChange}
      isDisabled={isDisabled}
      onEvent={emitEvent}
      EObject={EObject}
      {...attribute.fieldProps}
      {...featureFlags}
    />
  );
};
