import React, { useState, useCallback, FC } from 'react';
import { Category } from 'components/Attribute2';
import { Event } from 'types/Event';
import { idNameToIdText } from '../idNameToIdText';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';
import { CategorizationInitialize, IdNameCategory } from './Categorization.types';

export const Categorization: FC<AttributeComponentProps & {
  onInitialize: CategorizationInitialize;
  url?: string;
}> = (props) => {
  const { attribute, onChange, isDisabled, onInitialize, onEvent, url = '' } = props;
  const value = (props.value as IdNameCategory) || null;
  const [isReadingLoading, setReadingLoading] = useState(false);
  const [isEditing, setEditing] = useState(false);

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const handleChange = useCallback(
    async (newValue: { id: number } | null) => {
      setReadingLoading(true);
      setEditing(false);
      try {
        await onChange(attribute, newValue && newValue.id);
      } finally {
        setReadingLoading(false);
      }
    },
    [onChange],
  );

  const handleInitialize = useCallback(() => {
    return onInitialize({
      value,
      endpoint: url,
      onChange: handleChange,
    });
  }, [onInitialize, value, handleChange]);

  return (
    <Category
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      isReadingLoading={isReadingLoading}
      readingValue={value && idNameToIdText(value)}
      editingValue={value && idNameToIdText(value)}
      onInitialize={handleInitialize}
      isDisabled={isDisabled}
      onEvent={emitEvent}
    />
  );
};
