import React, { useState, useCallback, useMemo, FC } from 'react';
import { get } from 'entry/api/common';
import { Skills as SkillsAttribute, SkillsEditingValue } from 'components/Attribute2';
import { Event } from 'types/Event';
import { idNameToIdText } from '../idNameToIdText';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';
import { IdNameSkill } from './Skills.types';

export const Skills: FC<AttributeComponentProps> = (props) => {
  const { attribute, onChange, isDisabled, onEvent } = props;
  const rawValue = (props.value as IdNameSkill[]) || [];
  const [isEditing, setEditing] = useState(false);

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const value = useMemo(
    () =>
      rawValue.map((skill) => ({
        ...idNameToIdText(skill),
        value: skill.value,
      })),
    [rawValue],
  );

  const handleChange = useCallback(
    async (newSkills: SkillsEditingValue) => {
      await onChange(attribute, newSkills);
    },
    [onChange],
  );

  const handleLoad = async (text: string) => {
    const response = await get({
      url: '/view/dicts/skills',
      data: {
        text,
      },
    });

    return (response.items || []).map(idNameToIdText);
  };

  return (
    <SkillsAttribute
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      readingValue={value}
      editingValue={value}
      onChange={handleChange}
      isDisabled={isDisabled}
      onLoad={handleLoad}
      onEvent={emitEvent}
    />
  );
};
