import React, { useState, useCallback, useMemo, FC } from 'react';
import { get } from 'entry/api/common';
import { Event } from 'types/Event';
import { idNameToIdText } from '../idNameToIdText';
import { SuggestSelectEditingValue } from '@crm/components/dist/Attribute2/SuggestSelect/SuggestSelect.types';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';
import { IdName } from '@crm/components/dist/AttributesByScheme/Select';
import { SuggestSelect } from '@crm/components/dist/Attribute2/SuggestSelect';
import { EObject } from 'types/entities';

export const Suggest: FC<AttributeComponentProps & {
  EObject?: EObject;
  url?: string;
  canDelete?: boolean;
}> = (props) => {
  const { attribute, onChange, isDisabled, EObject, onEvent, url = '', canDelete } = props;
  const rawValue = (props.value as IdName) || null;
  const [isReadingLoading, setReadingLoading] = useState(false);
  const [isEditing, setEditing] = useState(false);

  const value = useMemo(() => rawValue && idNameToIdText(rawValue), [rawValue]);

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const handleChange = useCallback(
    async (newValue: SuggestSelectEditingValue) => {
      setReadingLoading(true);
      try {
        await onChange(attribute, newValue);
      } finally {
        setReadingLoading(false);
      }
    },
    [onChange],
  );

  const handleLoad = async (text: string) => {
    const eObj = EObject
      ? {
          eType: EObject.etype,
          eId: EObject.eid,
        }
      : null;

    const response = await get({
      url,
      data: {
        text,
        ...eObj,
      },
    });

    return response.items.map(idNameToIdText);
  };

  return (
    <SuggestSelect
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      isReadingLoading={isReadingLoading}
      readingValue={value}
      editingValue={value}
      onChange={handleChange}
      isDisabled={isDisabled}
      onLoad={handleLoad}
      onEvent={emitEvent}
      canDelete={canDelete}
    />
  );
};
