import React, { useState, useCallback, FC } from 'react';
import { get } from 'entry/api/common';
import { Tags as TagsAttribute, TagsEditingValue } from 'components/Attribute2';
import escapeRegExp from 'lodash/escapeRegExp';
import { Event } from '@crm/components/dist/types/Event';
import { IdNameTagToIdTextTag } from './Tags.utils';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';
import { IdNameTag } from './Tags.types';

export const Tags: FC<AttributeComponentProps & {
  url?: string;
}> = (props) => {
  const { attribute, onChange, isDisabled, onEvent, url = '' } = props;
  const [isReadingLoading, setReadingLoading] = useState(false);
  const [isEditing, setEditing] = useState(false);
  const value = ((props.value as IdNameTag[]) || []).map(IdNameTagToIdTextTag);

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const handleLoad = async (text: string) => {
    const response = await get({
      url,
    });

    return response.items
      .map(IdNameTagToIdTextTag)
      .filter((tag) => new RegExp(`(${escapeRegExp(text)})`, 'gim').test(tag.text));
  };

  const handleChange = useCallback(
    async (newValue: TagsEditingValue) => {
      setReadingLoading(true);
      try {
        await onChange(attribute, newValue);
      } finally {
        setReadingLoading(false);
      }
    },
    [onChange],
  );

  return (
    <TagsAttribute
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      isReadingLoading={isReadingLoading}
      readingValue={value}
      editingValue={value}
      isDisabled={isDisabled}
      onChange={handleChange}
      onLoad={handleLoad}
      onEvent={emitEvent}
    />
  );
};
