import React, { useState, useCallback, useMemo, FC } from 'react';
import { Timer, TimerSubmitValues, TimerEditingValue } from 'components/Attribute2';
import { Event } from 'types/Event';
import { stopEditingEvent } from 'utils/events';
import { formatAddValues } from './Timers.utils';
import { AttributeComponentProps } from '@crm/components/dist/AttributesByScheme';

export const Timers: FC<AttributeComponentProps> = (props) => {
  const { attribute, value: rawValue = [], onChange, isDisabled, onEvent } = props;
  const [isReadingLoading, setReadingLoading] = useState(false);
  const [isEditing, setEditing] = useState(false);
  const rawSingleValue = ((rawValue as TimerSubmitValues) || [])[0] || null;
  const value = useMemo(
    () => rawSingleValue && { ...rawSingleValue, activationDate: rawSingleValue.startDate },
    [rawSingleValue],
  );

  const emitEvent = useCallback(
    (event: Event) => {
      onEvent?.(attribute, event);
    },
    [onEvent],
  );

  const changeValue = useCallback(
    async (newValue: TimerEditingValue) => {
      setReadingLoading(true);
      setEditing(false);
      emitEvent(stopEditingEvent());
      try {
        await onChange(attribute, newValue);
      } finally {
        setReadingLoading(false);
      }
    },
    [onChange, emitEvent],
  );

  const handleSubmit = useCallback(
    async (values: TimerSubmitValues) => {
      changeValue(formatAddValues(values));
    },
    [changeValue],
  );

  return (
    <Timer
      label={attribute.label}
      name={attribute.name}
      isEditing={isEditing}
      onEditingChange={setEditing}
      readingValue={value}
      editingValue={value}
      isReadingLoading={isReadingLoading}
      onSubmit={handleSubmit}
      onChange={changeValue}
      isDisabled={isDisabled}
      onEvent={emitEvent}
    />
  );
};
