import Icon from '@crm/components/dist/lego2/Icon';
import Spin from '@crm/components/dist/lego2/Spin';
import React, { useEffect, useRef, useState } from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import { TimeProgress } from './TimeProgress';
import { AudioProps } from './Audio.types';
import css from './Audio.module.css';

const Audio: React.FC<AudioProps> = ({ src, error, onError, onRetry }) => {
  const audioRef = useRef<HTMLAudioElement>(null);

  const [currentTime, setCurrentTime] = useState(0);
  const [duration, setDuration] = useState(0);
  const [playRate, setPlayRate] = useState(1);
  const [isPlaying, setPlaying] = useState(false);
  const isDisablePlaying = useRef(false);
  const [isLoading, setLoading] = useState(true);
  const [hasInnerError, setHasInnerError] = useState(false);
  const { message = 'Произошла ошибка', canRetry = true } = error || {};

  useEffect(() => {
    setLoading(true);
  }, [src]);

  useEffect(() => {
    if (!error) {
      setHasInnerError(false);
    }
  }, [error]);

  const onLoadedData = () => {
    setPlaying(false);
    setLoading(false);
    setHasInnerError(false);
    if (!audioRef.current) {
      return;
    }
    setDuration(audioRef.current.duration);
  };

  const onEnded = () => {
    setPlaying(false);
  };

  const onTimeUpdate = () => {
    if (!audioRef.current) {
      return;
    }
    setCurrentTime(audioRef.current.currentTime);
  };

  const handleRetryClick = () => {
    let doReloading = true;
    if (onRetry) {
      const returnedValue = onRetry();
      if (returnedValue != null) {
        doReloading = returnedValue;
      }
    }

    if (doReloading) {
      audioRef.current!.load();
    }
  };

  const togglePlayRate = () => {
    if (!audioRef.current || isDisablePlaying.current) {
      return;
    }
    let newPlayRate = playRate >= 2 ? 1 : playRate + 0.5;
    audioRef.current.playbackRate = newPlayRate;
    setPlayRate(newPlayRate);
  };

  const togglePlaybackStatus = () => {
    if (!audioRef.current || isDisablePlaying.current) {
      return;
    }

    if (isPlaying) {
      audioRef.current.pause();
    } else {
      isDisablePlaying.current = true;
      audioRef.current
        .play()
        .catch(() => {
          setPlaying(false);
        })
        .finally(() => {
          isDisablePlaying.current = false;
        });
    }
    setPlaying((play) => !play);
  };

  const setTime = (seconds) => {
    if (!audioRef.current) {
      return;
    }

    const formattedCurrentTime = Math.floor(seconds);

    setCurrentTime(formattedCurrentTime);
    audioRef.current.currentTime = formattedCurrentTime;
  };

  const handleAudioError = () => {
    if (onError) {
      onError();
    }
    // на случай если onError callback не обработал ошибку
    setHasInnerError(true);
    setLoading(false);
  };

  const showSpinner = isLoading && !error;
  const showPlayer = !isLoading && !hasInnerError && !error;
  const showError = error || (!isLoading && hasInnerError);

  return (
    <div>
      <audio
        src={src}
        hidden
        ref={audioRef}
        data-testid="audio"
        onLoadedData={onLoadedData}
        onEnded={onEnded}
        onTimeUpdate={onTimeUpdate}
        onError={handleAudioError}
      />
      {showSpinner && <Spin data-testid="loading" progress view="default" size="m" />}
      {showPlayer && (
        <div className={css.Audio}>
          <button data-testid="btn-play" className={css.Button} onClick={togglePlaybackStatus}>
            {isPlaying ? (
              <Icon className={css.Button__icon} svg="pause" />
            ) : (
              <Icon className={cx(css.Button__icon, css.Button__icon_play)} svg="play" />
            )}
          </button>
          <button
            data-testid="btn-rate"
            className={cx(css.Button, css.Button__icon_playrate)}
            onClick={togglePlayRate}
          >
            {`${playRate}x`}
          </button>
          <TimeProgress currentTime={currentTime} duration={duration} setTime={setTime} />
        </div>
      )}
      {showError && (
        <div className={css.Audio__error}>
          {message}
          {canRetry && (
            <Button className={css.Audio__retryButton} onClick={handleRetryClick}>
              Попробовать еще раз
            </Button>
          )}
        </div>
      )}
    </div>
  );
};

export { Audio };
