import React, { ComponentType, FC, useEffect, useState } from 'react';
import { get } from 'api/common';
import { AudioProps } from '../../Audio.types';
import { WithMetaPreloadingProps } from './withMetaPreloading.types';
import { canInterpretError, interpretError, internalError } from './withMetaPreloading.utils';

export const withMetaPreloading = <P extends AudioProps>(
  Audio: ComponentType<P>,
): FC<WithMetaPreloadingProps<P>> => (props) => {
  const { metaSrc, error: outerError, onError, src: outerSrc, onRetry, ...restProps } = props;
  const [error, setError] = useState(outerError);
  const [isMetaValid, setMetaValid] = useState(false);
  const src = isMetaValid ? outerSrc : undefined;

  const fetchMetaAndInterpretError = (metaSrc: string | undefined) => {
    if (!metaSrc) {
      setMetaValid(true);
      return;
    }

    get({
      url: metaSrc,
      showBackendErrorForUser: false,
    })
      .then(() => {
        setError(outerError);
        setMetaValid(true);
      })
      .catch((requestError) => {
        if (!canInterpretError(requestError)) {
          setError(internalError());
          return;
        }

        setError(interpretError(requestError));
      });
  };

  useEffect(() => {
    fetchMetaAndInterpretError(metaSrc);
  }, [metaSrc]);

  const handleError = () => {
    setError({
      canRetry: false,
    });
  };

  const handleRetry = () => {
    fetchMetaAndInterpretError(metaSrc);
    return false;
  };

  return (
    <Audio
      error={error}
      onError={handleError}
      onRetry={handleRetry}
      src={src}
      {...(restProps as P)}
    />
  );
};
