import React, { PureComponent } from 'react';
import { Audio as DesignAudio } from 'components/design/Audio';
import { AudioProps, AudioState } from './Audio.types';

export class Audio extends PureComponent<AudioProps, AudioState> {
  static defaultProps = {
    isHidden: false,
    canInteractWithAudio: true,
  };

  public constructor(props: AudioProps) {
    super(props);
    this.state = {
      currentTime: 0,
      duration: 0,
      isPlaying: false,
      playRate: 1,
    };
  }

  private isAudioLoaded(): boolean {
    const { audio } = this.props;
    return audio.readyState === 4;
  }

  private syncStateWithAudio(audio: HTMLAudioElement) {
    this.setState({
      currentTime: audio.currentTime,
      duration: audio.duration,
      isPlaying: !audio.paused,
      playRate: audio.playbackRate,
    });
  }

  private resetState() {
    this.setState({
      currentTime: 0,
      duration: 0,
      playRate: 1,
      isPlaying: false,
    });
  }

  private handleLoadedData = () => {
    this.setState({
      duration: this.props.audio.duration,
    });
  };

  private handleEnded = () => {
    this.setState({
      isPlaying: false,
    });
  };

  private handleTimeUpdate = () => {
    this.setState({
      currentTime: this.props.audio.currentTime,
    });
  };

  private handleRateChange = (playRate: number) => {
    if (this.props.isRateDisabled || !this.props.canInteractWithAudio) {
      return;
    }

    this.setState({
      playRate,
    });
    this.props.audio.playbackRate = playRate;
  };

  private handleTimeChange = (time: number) => {
    if (!this.props.canInteractWithAudio) {
      return;
    }

    const roundedTime = Math.floor(time);

    this.setState({
      currentTime: roundedTime,
    });
    this.props.audio.currentTime = roundedTime;
  };

  private handlePlayChange = () => {
    if (this.props.onPlayClick) {
      this.props.onPlayClick();
      return;
    }

    const { audio } = this.props;
    if (audio.paused) {
      audio.play();
    } else {
      audio.pause();
    }
  };

  private handlePlay = () => {
    this.setState({
      isPlaying: true,
    });
  };

  private handlePause = () => {
    this.setState({
      isPlaying: false,
    });
  };

  private subscribeOnAudioEvents() {
    const { audio } = this.props;

    audio.addEventListener('play', this.handlePlay);
    audio.addEventListener('pause', this.handlePause);
    audio.addEventListener('loadeddata', this.handleLoadedData);
    audio.addEventListener('ended', this.handleEnded);
    audio.addEventListener('timeupdate', this.handleTimeUpdate);
  }

  private unsubscribeFromAudioEvents() {
    const { audio } = this.props;

    audio.removeEventListener('play', this.handlePlay);
    audio.removeEventListener('pause', this.handlePause);
    audio.removeEventListener('loadeddata', this.handleLoadedData);
    audio.removeEventListener('ended', this.handleEnded);
    audio.removeEventListener('timeupdate', this.handleTimeUpdate);
  }

  public componentDidMount() {
    const { audio, canInteractWithAudio } = this.props;

    if (canInteractWithAudio) {
      this.subscribeOnAudioEvents();
      if (this.isAudioLoaded()) {
        this.syncStateWithAudio(audio);
      }
    }
  }

  public componentDidUpdate(prevProps: AudioProps) {
    if (!prevProps.canInteractWithAudio && this.props.canInteractWithAudio) {
      this.subscribeOnAudioEvents();
      if (this.isAudioLoaded()) {
        this.syncStateWithAudio(this.props.audio);
      }
    }

    if (prevProps.canInteractWithAudio && !this.props.canInteractWithAudio) {
      this.unsubscribeFromAudioEvents();
      this.resetState();
    }
  }

  public componentWillUnmount() {
    this.unsubscribeFromAudioEvents();
  }

  public render() {
    const { isHidden, isLoading, isRateDisabled, isPlayDisabled, children } = this.props;
    const { isPlaying, currentTime, duration, playRate } = this.state;

    return (
      <div>
        {!isHidden && (
          <DesignAudio
            isPlaying={isPlaying}
            onPlayChange={this.handlePlayChange}
            isPlayDisabled={isPlayDisabled}
            isRateDisabled={isRateDisabled}
            isLoading={isLoading}
            rate={playRate}
            onRateChange={this.handleRateChange}
            currentTime={currentTime}
            onCurrentTimeChange={this.handleTimeChange}
            duration={duration}
          >
            {children}
          </DesignAudio>
        )}
      </div>
    );
  }
}
