import React, { PureComponent } from 'react';
import Bluebird from 'bluebird';
import { get } from 'api/common';
import { Audio } from 'components/Audio2';
import { getAbsoluteUrl } from 'utils/getAbsoluteUrl';
import { ErrorMessage } from './ErrorMessage';
import { AudioSourceProps, AudioSourceState } from './AudioSource.types';
import {
  canInterpretError,
  internalError,
  interpretError,
  metaObjectFromError,
  okMetaObject,
  META_OK_STATUS,
} from './AudioSource.utils';
import { MetaCache } from './MetaCache';

export class AudioSource extends PureComponent<AudioSourceProps, AudioSourceState> {
  private lastMetaRequest: Bluebird<unknown> | null = null;

  private isPlayingDisabled = false;

  private metaCache = MetaCache.create();

  static defaultProps = {
    keepPlayOnUnmount: true,
  };

  public constructor(props: AudioSourceProps) {
    super(props);
    this.state = {
      canAudioInteract: this.isAudioSrcSame(),
      error: null,
      isMetaValid: false,
      isMetaLoading: false,
    };
  }

  private isAudioSrcSame(): boolean {
    const { audio, src } = this.props;
    if (!src) {
      return false;
    }

    return getAbsoluteUrl(src) === audio.src;
  }

  private togglePlaying = () => {
    const { audio, src } = this.props;
    const { isMetaValid } = this.state;
    if (this.isPlayingDisabled || src == null) {
      return;
    }

    if (!isMetaValid) {
      this.fetchMetaAndInterpretError();
      return;
    }

    if (!this.isAudioSrcSame()) {
      audio.src = src;
      audio.load();
      this.setState({
        canAudioInteract: true,
      });
    }

    if (audio.paused) {
      this.isPlayingDisabled = true;
      audio
        .play()
        .catch(() => {
          audio.pause();
        })
        .finally(() => {
          this.isPlayingDisabled = false;
        });
    } else {
      audio.pause();
    }
  };

  private setStateForOkMeta(callback?: () => void) {
    this.setState(
      {
        isMetaValid: true,
        isMetaLoading: false,
        error: null,
      },
      callback,
    );
  }

  private fetchMetaAndInterpretError = (successCallback?: () => void) => {
    const { metaSrc } = this.props;

    if (!metaSrc) {
      this.setState({
        isMetaValid: true,
        error: null,
      });
      this.lastMetaRequest = null;
      return;
    }

    if (this.lastMetaRequest) {
      this.lastMetaRequest.cancel();
    }

    const cachedMeta = this.metaCache.get(metaSrc);
    if (cachedMeta) {
      if (cachedMeta.status === META_OK_STATUS) {
        this.setStateForOkMeta(successCallback);
        return;
      }

      this.setState({
        error: interpretError(cachedMeta),
      });
      return;
    }

    this.setState({
      isMetaLoading: true,
    });
    this.lastMetaRequest = get({
      url: metaSrc,
      showBackendErrorForUser: false,
    })
      .then(() => {
        this.metaCache.set(metaSrc, okMetaObject());
        this.setStateForOkMeta(successCallback);
      })
      .catch((requestError) => {
        this.metaCache.set(metaSrc, metaObjectFromError(requestError));
        if (!canInterpretError(requestError)) {
          this.setState({
            error: internalError(),
          });
          return;
        }

        this.setState({
          error: interpretError(requestError),
        });
      })
      .finally(() => {
        this.setState({
          isMetaLoading: false,
        });
      });

    return this.lastMetaRequest;
  };

  private handleRetry = () => {
    if (this.props.metaSrc) {
      this.metaCache.remove(this.props.metaSrc);
    }
    this.fetchMetaAndInterpretError(this.togglePlaying);
  };

  private handleLoadedData = () => {
    if (!this.isAudioSrcSame()) {
      this.setState({
        canAudioInteract: false,
      });
    }
  };

  public componentDidMount() {
    const { audio, metaCacheClearKey } = this.props;

    audio.addEventListener('loadeddata', this.handleLoadedData);
    if (metaCacheClearKey != null) {
      this.metaCache.keepCacheValidForClearKey(metaCacheClearKey);
    }
    this.fetchMetaAndInterpretError();
  }

  public componentWillUnmount() {
    const { audio, keepPlayOnUnmount } = this.props;

    if (this.lastMetaRequest) {
      this.lastMetaRequest.cancel();
    }

    if (!keepPlayOnUnmount && this.isAudioSrcSame()) {
      audio.pause();
    }

    audio.removeEventListener('loadeddata', this.handleLoadedData);
  }

  public render() {
    const { audio } = this.props;
    const { error, canAudioInteract, isMetaValid, isMetaLoading } = this.state;
    const { message = 'Произошла ошибка' } = error || {};

    return (
      <Audio
        audio={audio}
        onPlayClick={this.togglePlaying}
        canInteractWithAudio={canAudioInteract}
        isLoading={isMetaLoading}
        isPlayDisabled={!isMetaValid}
        isRateDisabled={!isMetaValid}
      >
        {error && <ErrorMessage message={message} />}
      </Audio>
    );
  }
}
