import { ErrorObject } from './AudioSource.types';
import { MetaObject } from './MetaCache/MetaCache.types';

export const RECORD_NEVER_EXISTS_STATUS = 410;
export const RECORD_NOT_READY_YET_STATUS = 404;
export const INTERNAL_ERROR_STATUS = 500;
export const META_OK_STATUS = 200;

export const doesRecordNeverExists = (error: { status: number }): boolean => {
  return error.status === RECORD_NEVER_EXISTS_STATUS;
};

export const recordNeverExistsError = (message?: string): ErrorObject => ({
  message: message || 'Запись отсутствует',
  canRetry: false,
});

export const isRecordNotReadyYet = (error: { status: number }): boolean => {
  return error.status === RECORD_NOT_READY_YET_STATUS;
};

export const notReadyRecordError = (_message?: string): ErrorObject => ({
  message: 'Запись еще не получена',
  canRetry: true,
});

export const canInterpretError = (error: unknown): boolean => {
  return (
    Boolean(error && (error as { status?: number }).status) &&
    Boolean((error as { data?: string | unknown }).data) &&
    typeof (error as { data?: string | unknown }).data === 'string'
  );
};

export const internalError = (message?: string): ErrorObject => ({
  message: message || 'Произошла ошибка',
  canRetry: true,
});

export const interpretError = (error: { data?: string; status: number }): ErrorObject => {
  if (doesRecordNeverExists(error)) {
    return recordNeverExistsError(error.data);
  }

  if (isRecordNotReadyYet(error)) {
    return notReadyRecordError(error.data);
  }

  return internalError(error.data);
};

export const recordNeverExistsMetaObject = () => ({
  status: RECORD_NEVER_EXISTS_STATUS,
});

export const recordNotReadyYetMetaObject = () => ({
  status: RECORD_NOT_READY_YET_STATUS,
});

export const internalErrorMetaObject = () => ({
  status: INTERNAL_ERROR_STATUS,
});

export const okMetaObject = () => ({
  status: META_OK_STATUS,
});

export const metaObjectFromError = (error: { data?: string; status: number }): MetaObject => {
  if (doesRecordNeverExists(error)) {
    return recordNeverExistsMetaObject();
  }

  if (isRecordNotReadyYet(error)) {
    return recordNotReadyYetMetaObject();
  }

  return internalErrorMetaObject();
};

export const errorFromMetaObject = (metaObject: MetaObject): ErrorObject => {
  return interpretError(metaObject);
};
