import React, { useState, CSSProperties } from 'react';
import cx from 'classnames';
import Icon from '@crm/components/dist/lego2/Icon';
import { User } from 'types/entities/user';
import { SuggestUser } from 'types/SuggestUser';
import { ExternalPassportUser } from 'types/ExternalPassportUser';
import { isExternalUser } from 'utils/isExternalUser';
import { getLogin } from 'utils/getLogin';
import { getSign } from 'utils/getSign';
import css from './Avatar.module.css';
import getFallbackStyles from './getFallbackStyles';
import { useAvatarService } from './useAvatarService';

export type AvatarUser = User | ExternalPassportUser | SuggestUser | undefined;

export interface AvatarProps {
  login?: string; // deprecated: use user prop
  className?: string;
  theme?: string;
  size?: string;
  user: AvatarUser;
  style?: CSSProperties;
}

export interface UserOrLoginData {
  user?: AvatarUser;
  login?: string;
}

const Avatar: React.FC<AvatarProps> = (props) => {
  const {
    user,
    login,
    className,
    theme,
    size = 'customL',
    style,
    // https://st.yandex-team.ru/CRM-15295
    contentPlaceholder,
    zIndex,
    ...other
    // eslint-disable-next-line
  } = props as any;

  const classNameProp = cx(className, css[`b_theme_${theme}`], css[`b_size_${size}`]);

  const userLogin = getLogin({ user, login });

  const { srcSet, isServiceAvailable } = useAvatarService(user, login);

  if (!user && !login) {
    return (
      <span style={style} className={cx(css.b_empty, classNameProp)}>
        <Icon className={css.b_empty__icon} svg="person" />
      </span>
    );
  }

  if (!isServiceAvailable || (isExternalUser(user) && !user.avatarId) || !user.login) {
    return (
      <span
        className={cx(css.b_fallback, classNameProp)}
        style={{ ...style, ...getFallbackStyles(userLogin) }}
      >
        <span className={css.b__text}>{getSign({ user, login })}</span>
      </span>
    );
  }

  return (
    <img
      src={srcSet[0]}
      srcSet={srcSet.join(', ')}
      alt="avatar"
      style={style}
      {...other}
      className={classNameProp}
    />
  );
};

export const AvatarWithUpdate: React.FC<AvatarProps> = (props) => {
  const { user } = props;

  const { service } = useAvatarService(user);
  const [, updateState] = useState<boolean>();

  service.setOnAvailableChange(updateState);

  return <Avatar {...props} />;
};

export default Avatar;
