type AvailableChangeHandler = (flag: boolean) => void;

class BaseAvatarService {
  private img: HTMLImageElement | null = null;

  private onAvailableChange: AvailableChangeHandler | null = null;

  constructor(private testSrc?: string) {}

  public isAvailable = false;

  public setOnAvailableChange(func: AvailableChangeHandler | null) {
    this.onAvailableChange = func;
  }

  public checkStatus() {
    if (!this.testSrc) {
      return;
    }

    this.img = document.createElement('img');
    this.img.src = this.testSrc;
    this.img.hidden = true;
    this.img.onload = () => {
      this.setAvailable(true);
      this.cleanup();
    };
    this.img.onerror = () => {
      this.setAvailable(false);
      this.cleanup();
    };

    window.setTimeout(() => {
      this.cleanup();
    }, 500);
    document.body.append(this.img);
  }

  private cleanup() {
    if (this.img) {
      this.img.remove();
      this.img = null;
    }
  }

  public setAvailable(flag: boolean) {
    if (flag === this.isAvailable) {
      return;
    }

    this.isAvailable = flag;
    if (this.onAvailableChange) {
      this.onAvailableChange(flag);
    }
  }
}

class AvatarService extends BaseAvatarService {
  constructor() {
    super('//center.yandex-team.ru/api/v1/user/robot-crm/avatar/64.jpg');
  }

  public getSrcSet(userLogin?: string) {
    return [
      `//center.yandex-team.ru/api/v1/user/${userLogin}/avatar/64.jpg`,
      `//center.yandex-team.ru/api/v1/user/${userLogin}/avatar/128.jpg 2x`,
    ];
  }
}

export const avatarService = new AvatarService();

class ExternalAvatarService extends BaseAvatarService {
  constructor() {
    super();
    this.isAvailable = true;
  }

  public getSrcSet(avatarId?: string) {
    return [
      `//avatars.mds.yandex.net/get-yapic/${avatarId}/64.jpg`,
      `//avatars.mds.yandex.net/get-yapic/${avatarId}/128.jpg 2x`,
    ];
  }
}

export const externalAvatarService = new ExternalAvatarService();
