import React, {
  useEffect,
  useRef,
  useCallback,
  useMemo,
  KeyboardEvent as ReactKeyboardEvent,
  Fragment,
} from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { BubblesContainer } from 'components/BubblesContainer';
import { useKeyboard } from 'utils/hooks/useKeyboard';
import { Bubble, BubbleProps } from 'components/Bubble';
import throttle from 'lodash/throttle';
import { ContentEditable } from './ContentEditable';
import { BubbleInputProps, Bubble as IBubble, RenderBubble } from './BubbleInput.types';
import css from './BubbleInput.module.css';
import { CARET_X_THRESHOLD, CARET_Y_THRESHOLD, DELETE_THROTTLE_DELAY } from './BubbleInput.config';

const crossIconProvider = (cls) => <Icon className={cls} svg="close2" />;
const defaultBubbleRender = <T extends {} = IBubble>(bubble: T, props: BubbleProps) => (
  <Bubble {...props}>{((bubble as unknown) as IBubble).text}</Bubble>
);
const defaultGetKey = <T extends {}>(bubble: T) => ((bubble as unknown) as IBubble).id;

export const BubbleInput = <T extends {} = IBubble>(props: BubbleInputProps<T>) => {
  const {
    text = '',
    onTextChange,
    tabIndex,
    hasClear = true,
    bubbles = [],
    onBubbleDelete,
    renderBubble = defaultBubbleRender as RenderBubble<T>,
    getKey = defaultGetKey,
    placeholder,
    autoFocus = true,
    innerRef,
    contentEditableRef: outerContentEditableRef,
    display = true,
    isMultiline = true,
    canUseKeyboard = false,
    onFocusedIndexChange,
    isContentEditableDisabled,
    canDelete = true,
  } = props;

  const focusedIndexRef = useRef(-1);
  const rootRef = useRef<HTMLDivElement>(null);
  const contentEditableRef = useRef<HTMLDivElement>();

  const throttledBubbleDelete = useCallback(
    throttle((value: T) => onBubbleDelete?.(value), DELETE_THROTTLE_DELAY),
    [onBubbleDelete],
  );

  const transitionToEditor = () => {
    setTimeout(() => contentEditableRef.current?.focus(), 0);
  };

  const changeFocus = useCallback(
    (indexOrCallback: number | ((prev: number) => number)) => {
      if (typeof indexOrCallback === 'number') {
        focusedIndexRef.current = indexOrCallback;
      } else {
        focusedIndexRef.current = indexOrCallback(focusedIndexRef.current);
      }

      if (focusedIndexRef.current !== -1) {
        (rootRef.current?.querySelectorAll('[data-bubble-root]')[
          focusedIndexRef.current
        ] as HTMLElement)?.focus();
      } else {
        transitionToEditor();
      }
      onFocusedIndexChange?.(focusedIndexRef.current);
    },
    [onFocusedIndexChange],
  );

  const isSelectionInEditor = (selection: Selection) => {
    return (
      selection.anchorNode === contentEditableRef.current ||
      selection.anchorNode?.parentElement === contentEditableRef.current
    );
  };

  const handleChange = useCallback((newValue: string) => {
    onTextChange?.(newValue);
    if (newValue) {
      focusedIndexRef.current = -1;
    }
  }, []);

  const handleKeyDown = useCallback((event: ReactKeyboardEvent) => {
    if (event.key === 'Enter') {
      event.preventDefault();
    }
  }, []);

  const handleCrossClick = useCallback(() => {
    const contentEditable = contentEditableRef.current!;
    if (onTextChange) {
      onTextChange('');
      contentEditable.focus();
    }
  }, [onTextChange]);

  useEffect(() => {
    if (innerRef) {
      innerRef(rootRef.current!);
    }
  }, [innerRef]);

  useEffect(() => {
    if (outerContentEditableRef) {
      outerContentEditableRef(contentEditableRef.current!);
    }
  }, [outerContentEditableRef]);

  useEffect(() => {
    const contentEditable = contentEditableRef.current!;
    const root = rootRef.current!;

    const handleFocusOut = (event: FocusEvent) => {
      const { target, relatedTarget } = event;

      if (!(target as HTMLElement)?.dataset.bubbleRoot) {
        return;
      }

      if ((relatedTarget as HTMLElement)?.dataset.bubbleRoot) {
        return;
      }

      changeFocus(-1);
    };

    const handleRootMouseDown = (event: MouseEvent) => {
      if (event.target !== root) {
        return;
      }

      // пытаемся поставить курсор к ближайшему символу
      const clientRect = contentEditable.getBoundingClientRect();
      const clientX = event.clientX;
      const clientY = event.clientY;
      let finalX = Math.max(
        Math.min(clientRect.left + clientRect.width - CARET_X_THRESHOLD, clientX),
        clientRect.left + CARET_X_THRESHOLD,
      );
      let finalY = Math.max(
        Math.min(clientRect.top + clientRect.height - CARET_Y_THRESHOLD, clientY),
        clientRect.top + CARET_Y_THRESHOLD,
      );

      const range = document.createRange();
      if (document.caretRangeFromPoint) {
        const caretPosition = document.caretRangeFromPoint(finalX, finalY);
        range.setStart(caretPosition.startContainer, caretPosition.startOffset);
        range.setEnd(caretPosition.endContainer, caretPosition.endOffset);
      } else {
        const caretPosition = document.caretPositionFromPoint(finalX, finalY);
        if (!caretPosition) {
          return;
        }

        range.setStart(caretPosition.offsetNode, caretPosition.offset);
        range.setEnd(caretPosition.offsetNode, caretPosition.offset);
      }

      const selection = window.getSelection();
      if (!selection) {
        return;
      }

      range.collapse(false);
      selection.removeAllRanges();
      selection.addRange(range);
      event.preventDefault();
    };
    root.addEventListener('mousedown', handleRootMouseDown);
    root.addEventListener('focusout', handleFocusOut);

    return () => {
      root.removeEventListener('mousedown', handleRootMouseDown);
      root.removeEventListener('focusout', handleFocusOut);
    };
  }, [changeFocus]);

  const transitionToLeft = useCallback(() => {
    contentEditableRef.current?.blur();
    getSelection()?.removeAllRanges();
    changeFocus((prev) => {
      if (prev === -1) {
        return bubbles.length - 1;
      }
      return Math.max(prev - 1, 0);
    });
  }, [bubbles, changeFocus]);

  const transitionToRight = useCallback(() => {
    changeFocus((prev) => {
      if (prev === -1) {
        return 0;
      }

      return Math.min(prev + 1, bubbles.length - 1);
    });
  }, [bubbles, changeFocus]);

  const handleBackspace = useCallback(
    (event: KeyboardEvent) => {
      if (!bubbles.length) {
        return;
      }

      const selection = getSelection();
      if (selection && isSelectionInEditor(selection)) {
        if (selection?.anchorOffset === 0 && selection.isCollapsed) {
          transitionToLeft();
        }
        return;
      }

      const focusedIndex = focusedIndexRef.current;
      const focusedBubble = bubbles[focusedIndex];

      if (!event.repeat) {
        throttledBubbleDelete.cancel();
      }

      changeFocus((prev) => prev - 1);
      throttledBubbleDelete?.(focusedBubble);
    },
    [bubbles, throttledBubbleDelete, changeFocus, transitionToLeft],
  );

  useKeyboard(
    {
      onTab: (event) => event.preventDefault(),
      onBackspace: handleBackspace,
      onUp: (event) => {
        event.preventDefault();
        transitionToEditor();
      },
      onDown: (event) => {
        event.preventDefault();
        transitionToEditor();
      },
      onLeft: () => {
        if (!bubbles.length) {
          return;
        }

        const selection = getSelection();
        if (selection && isSelectionInEditor(selection) && selection?.anchorOffset !== 0) {
          return;
        }

        transitionToLeft();
      },
      onRight: () => {
        if (!bubbles.length) {
          return;
        }

        const focusedIndex = focusedIndexRef.current;
        if (focusedIndex === bubbles.length - 1) {
          transitionToEditor();
          return;
        }

        const selection = getSelection();
        if (selection && isSelectionInEditor(selection)) {
          return;
        }

        transitionToRight();
      },
    },
    canUseKeyboard && display,
    [handleBackspace, bubbles, transitionToEditor, transitionToLeft, transitionToRight],
  );

  const isClearVisible = hasClear && text.length > 0;
  const className = cx(css.BubbleInput, {
    [css.BubbleInput_contentEditableDisabled]: isContentEditableDisabled,
    [css.BubbleInput_singleline]: !isMultiline,
  });

  const handleBubbleDelete = useCallback(
    (bubble: T) => {
      onBubbleDelete?.(bubble);
      contentEditableRef.current?.focus();
    },
    [onBubbleDelete],
  );

  const renderedBubbles = useMemo(
    () =>
      bubbles.map((bubble) => (
        <Fragment key={getKey(bubble)}>
          {renderBubble(bubble, {
            canDelete,
            onDelete: () => handleBubbleDelete(bubble),
          })}
        </Fragment>
      )),
    [bubbles, renderBubble, handleBubbleDelete, canDelete],
  );

  return (
    <BubblesContainer
      className={className}
      hidden={!display}
      innerRef={rootRef}
      data-testid="BubbleInput"
    >
      {renderedBubbles}
      <ContentEditable
        className={css.BubbleInput__contentEditable}
        display={display}
        autoFocus={autoFocus}
        isDisabled={isContentEditableDisabled}
        isMultiline={isMultiline}
        tabIndex={display ? tabIndex : -1}
        placeholder={bubbles.length ? '' : placeholder}
        onKeyDown={handleKeyDown}
        innerRef={(instance) => (contentEditableRef.current = instance)}
        value={text}
        onChange={handleChange}
      />
      {isClearVisible && (
        <Button
          view="clear"
          tabIndex={-1}
          className={css.BubbleInput__cross}
          icon={crossIconProvider}
          onClick={handleCrossClick}
        />
      )}
    </BubblesContainer>
  );
};
