import React, { FC, ComponentType, useRef, useEffect, useState } from 'react';
import cx from 'classnames';
import { config } from 'services/Config';
import {
  Calendar as BigCalendar,
  EventWrapperProps,
  View,
  Components,
} from '@crm/react-big-calendar';
import { Event } from './Event';
import { Event as IEvent } from './Event/Event.types';
import { CalendarProps } from './Calendar.types';
import '@crm/react-big-calendar/lib/css/react-big-calendar.css';
import './Calendar.css';
import css from './Calendar.module.css';
import { localizer } from './localizer';
import { adaptRange } from './utils/adaptRange';
import { context } from './eventContext';

const views: View[] = ['month', 'week', 'day'];
export const Calendar: FC<CalendarProps> = (props) => {
  const {
    events = [],
    onRangeChange,
    view = 'week',
    onView,
    date,
    onNavigate,
    toolbar,
    onEventEdit,
    onEventDelete,
    eventMinHeight = 1.8,
  } = props;
  const components: Components<IEvent> = {
    eventWrapper: ((props.components || {}).event || Event) as ComponentType<
      EventWrapperProps<IEvent>
    >,
  };
  const [scrollToTime, setScrollToTime] = useState<Date | undefined>();
  const wasCurrentRangeScrolled = useRef(false);

  const newActivitiesDesign = config.value.features.newActivitiesDesign;
  const className = cx(css.Calendar, props.className, {
    [css.Calendar_newDesign]: newActivitiesDesign,
  });

  useEffect(() => {
    if (!events.length) {
      const scrollDate = new Date(date.toISOString());
      scrollDate.setHours(9, 0, 0);
      setScrollToTime(scrollDate);
      return;
    }

    if (!wasCurrentRangeScrolled.current) {
      document.querySelector('[data-event]')?.scrollIntoView();
      wasCurrentRangeScrolled.current = true;
    }
  }, [events]);

  const handleRangeChange = (range, newView) => {
    wasCurrentRangeScrolled.current = false;
    if (onRangeChange) {
      onRangeChange(adaptRange(range, newView || view));
    }
  };

  return (
    <context.Provider
      value={{
        onEdit: onEventEdit,
        onDelete: onEventDelete,
      }}
    >
      <BigCalendar
        date={date}
        className={className}
        view={view}
        events={events}
        views={views}
        culture="ru"
        localizer={localizer}
        startAccessor="start"
        endAccessor="end"
        onSelectEvent={props.onEventSelect}
        onNavigate={onNavigate}
        onRangeChange={handleRangeChange}
        onView={onView}
        dayLayoutAlgorithm="no-overlap"
        eventMinHeight={eventMinHeight}
        toolbar={toolbar}
        scrollToTime={scrollToTime}
        // eslint-disable-next-line
        components={components as any}
        displayCurrentTime={newActivitiesDesign}
      />
    </context.Provider>
  );
};
