import React, { FC, useRef, useState, MouseEvent, useEffect, useContext } from 'react';
import cx from 'classnames';
import { config } from 'services/Config';
import Popup from '@crm/components/dist/lego2/Popup';
import { FormBySchemeLoadableService } from 'components/FormBySchemeLoadable/FormBySchemeLoadable.service';
import { FormBySchemePreview } from 'components/FormByScheme/FormBySchemePreview';
import { Text } from '@crm/components/dist/lego2/Text';
import { observer } from 'mobx-react-lite';
import { context } from '../eventContext';
import { EventProps } from './Event.types';
import css from './Event.module.css';
import { normalizeStyle } from './Event.utils';

export const Event: FC<EventProps> = observer((props) => {
  const { event, type } = props;
  const { editingUrl, start, disabled = false } = event;
  const { onEdit, onDelete } = useContext(context);

  const newActivitiesDesign = config.value.features.newActivitiesDesign;

  const [service, setService] = useState(() => new FormBySchemeLoadableService(editingUrl));

  useEffect(() => {
    setService(new FormBySchemeLoadableService(editingUrl));

    return () => {
      service.destroy();
    };
  }, [editingUrl]);
  const [isPopupVisible, setPopupVisible] = useState(false);

  const ref = useRef<HTMLDivElement>(null);
  const handleClick = async (event: MouseEvent<HTMLElement>) => {
    if (disabled) {
      return;
    }

    if (props.onClick) {
      props.onClick(event);
    }

    await service.loadForm();

    setPopupVisible(true);
  };

  const close = () => {
    setPopupVisible(false);
  };

  const handleEdit = () => {
    if (onEdit) {
      onEdit(event);
      setPopupVisible(false);
    }
  };

  const handleDelete = () => {
    if (onDelete && confirm('Удалить активность?')) {
      onDelete(event);
      setPopupVisible(false);
    }
  };

  const className = cx(css.Event, {
    [css.Event_type_date]: type === 'date',
    [css.Event_type_time]: type === 'time',
    [css.Event_newDesign]: newActivitiesDesign,
    [css.Event_platform_staff]: newActivitiesDesign && !event.id,
  });

  const style = {
    ...normalizeStyle(props.style),
    background: event.color,
  };

  if (newActivitiesDesign) {
    style.background = undefined;
  }

  const form = service.formData;

  return (
    <div
      data-event
      onClick={handleClick}
      onDoubleClick={props.onDoubleClick}
      className={className}
      style={style}
      ref={ref}
    >
      {newActivitiesDesign ? (
        <Text weight="medium" typography="caption-l">
          {event.title}
        </Text>
      ) : (
        <>
          <div>
            {start.toLocaleTimeString('ru', {
              hour: '2-digit',
              minute: '2-digit',
            })}
          </div>
          <span className={css.Event__title}>{event.title}</span>
        </>
      )}

      <Popup
        key={String(Boolean(form))}
        className={css.Event__popup}
        view="default"
        onOutsideClick={close}
        onClose={close}
        stopPropagation
        keepMounted={false}
        visible={isPopupVisible}
        anchor={ref}
        target="anchor"
      >
        {form && <FormBySchemePreview scheme={form} onEdit={handleEdit} onDelete={handleDelete} />}
      </Popup>
    </div>
  );
});
