import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import CalendarEventDate from './CalendarEventDate';
import CalendarEventParticipant from './CalendarEventParticipant';
import CalendarEventAttendees from './CalendarEventAttendees';
import css from './CalendarEvent.modules.scss';

const CalendarEvent = (props) => {
  const { event, className } = props;

  if (!event) {
    return null;
  }

  return (
    <div className={cx(className, css.CalendarEvent)}>
      {
        event.summary &&
        <div className={css.CalendarEvent__title}>
          {event.summary}
        </div>
      }
      <div className={css.CalendarEvent__meta}>
        <div className={css.CalendarEvent__row}>
          <CalendarEventDate
            startTime={event.startTime}
            startTimeZone={event.startTimeZone}
            endTime={event.endTime}
            endTimeZone={event.endTimeZone}
            duration={event.duration}
          />
        </div>
        {
          event.description &&
          <div className={css.CalendarEvent__row}>
            <span className={css.CalendarEvent__label}>Описание</span>
            <span
              className={cx(css.CalendarEvent__value, css.CalendarEvent__description)}
            >
              {event.description}
            </span>
          </div>
        }
        {
          event.organizer &&
          <div className={css.CalendarEvent__row}>
            <span className={css.CalendarEvent__label}>Организатор</span>
            <span className={css.CalendarEvent__value}>
              <CalendarEventParticipant {...event.organizer} />
            </span>
          </div>
        }
        {
          Array.isArray(event.attendees) &&
          <div className={css.CalendarEvent__row}>
            <span className={css.CalendarEvent__label}>Участники</span>
            <span className={css.CalendarEvent__value}>
              <CalendarEventAttendees attendees={event.attendees} />
            </span>
          </div>
        }
      </div>
    </div>
  );
};

CalendarEvent.propTypes = {
  className: PropTypes.string,
  event: PropTypes.shape({
    summary: PropTypes.string,
    description: PropTypes.string,
    startTime: PropTypes.string,
    startTimeZone: PropTypes.string,
    endTime: PropTypes.string,
    endTimeZone: PropTypes.string,
    isAllDay: PropTypes.bool,
    duration: PropTypes.string,
    organizer: PropTypes.shape(CalendarEventParticipant.propTypes),
    ...CalendarEventAttendees.propTypes,
  }),
};

CalendarEvent.defaultProps = {
  className: undefined,
  event: undefined,
};

export default CalendarEvent;
