import React, { useCallback } from 'react';
import {
  AttributesByScheme,
  GetAccountInput,
  GetOpportunitiesInput,
} from 'components/AttributesByScheme';
import { AttributesSchemeRow, ComponentsProperty } from '@crm/components/dist/AttributesByScheme';
import Access from 'utils/Access';
import { Opportunity } from 'components/Attribute2/types/Opportunity';
import { MutationData } from 'services/PersonalCallService/PersonalCallService.types';
import { CallAttributesProps, Tag } from './CallAttributes.types';
import { ACCOUNT_FEATURE_FLAGS } from './CallAttributes.config';

export const CallAttributes: React.FC<CallAttributesProps> = ({
  callData,
  onOpportunitiesChange,
  onTagsChange,
  onCommonChange,
}) => {
  const { scheme, data, props } = callData;

  const getCommon = useCallback(
    (attribute: AttributesSchemeRow) => {
      return {
        value: data[attribute.name] ?? undefined,
        isDisabled: !Access.isEdit(attribute.access || props[attribute.name]?.access),
      };
    },
    [data, props],
  );

  const getOpportunitiesInput = useCallback<GetOpportunitiesInput>(
    (attribute) => {
      return {
        ...getCommon(attribute),
        account: data.account,
      };
    },
    [data.account, getCommon],
  );

  const commonChange = useCallback(
    async (attribute: AttributesSchemeRow, value: unknown) => {
      const data: MutationData = {
        name: attribute.backendUpdateKey || attribute.name,
        operation: 'update',
        value,
      };
      return onCommonChange(data);
    },
    [onCommonChange],
  );

  const changeOpportunities = useCallback(
    (attribute, value: Opportunity[]) => {
      return onOpportunitiesChange(attribute, value);
    },
    [onOpportunitiesChange],
  );

  const changeTags = useCallback(
    (attribute, value: Tag[]) => {
      return onTagsChange(attribute, value);
    },
    [onTagsChange],
  );

  const accountAccesses = (props.account as unknown) as {
    accessNullAccount: number;
    accessCreateAccount: number;
    accessCreateLead: number;
  };

  const getAccountInput = useCallback<GetAccountInput>(
    (attribute) => {
      return {
        ...getCommon(attribute),
        featureFlags: {
          ...ACCOUNT_FEATURE_FLAGS,
          canShowNullClientButton: Access.isEdit(accountAccesses.accessNullAccount),
          canShowCreateButton: Access.isEdit(accountAccesses.accessCreateAccount),
          canShowLeadButton: Access.isEdit(accountAccesses.accessCreateLead),
        },
      };
    },
    [
      accountAccesses.accessCreateAccount,
      accountAccesses.accessCreateLead,
      accountAccesses.accessNullAccount,
      getCommon,
    ],
  );

  const components: ComponentsProperty = {
    Tags: {
      changeHandler: changeTags,
    },
    OpportunitiesInput: {
      getProps: getOpportunitiesInput,
      changeHandler: changeOpportunities,
    },
    AccountInput: {
      getProps: getAccountInput,
    },
  };

  return (
    <AttributesByScheme
      scheme={scheme}
      onCommonChange={commonChange}
      getCommon={getCommon}
      components={components}
    />
  );
};
