import React, { useCallback } from 'react';
import { Text } from '@crm/components/dist/lego2/Text';
import cx from 'classnames';
import xor from 'lodash/xor';
import Icon from '@crm/components/dist/lego2/Icon';
import Button from '@crm/components/dist/lego2/Button';
import Popup from '@crm/components/dist/lego2/Popup';
import Spin from '@crm/components/dist/lego2/Spin';
import { Timer } from 'components/Timer';
import { userInteractionService } from 'services/UserInteractionService';
import { IssueEntity } from 'types';
import createI18N from '@yandex-int/i18n';
import { usePersonalCallService } from 'services/PersonalCallService';
import { MutationData } from 'services/PersonalCallService/PersonalCallService.types';
import { observer } from 'mobx-react-lite';
import { PersonalCallData, PersonalCallObject } from '../../../types/PersonalCallData';
import css from './CallPopup.module.css';
import { CallPopupProps } from './CallPopup.types';
import { CallAttributes, Tag } from './CallAttributes';
import { getDeletedOpportunities, getNewOpportunities } from './CallPopup.utils';
import * as keyset from '../CallPopup.i18n';

const i18n = createI18N(keyset);
const i18nTitle = i18n('popupTitle');
const i18nDone = i18n('popupDone');
const i18nValidationAlert = i18n('validationAlert');

export const CallPopup: React.FC<CallPopupProps> = observer(
  ({ isVisible, onMinimize, isCallActive = false, isLoading = false }: CallPopupProps) => {
    const callService = usePersonalCallService();

    const { callData = {} as PersonalCallObject } = callService;

    const { id = 0, data = {} as PersonalCallData } = callData;

    const handleTagsChange = useCallback(
      async (attribute, newValue: Tag[]) => {
        const oldValue = data[attribute.name] || [];
        const oldIds = oldValue.map((item) => item.id);
        const newIds = newValue.map((item) => item.id);
        const operation = oldIds.length > newIds.length ? 'remove' : 'add';
        const tagId = xor(oldIds, newIds)[0];
        return callService.mutate(id, {
          name: attribute.name,
          operation,
          value: tagId,
        });
      },
      [callService, data, id],
    );

    const handleOpportunitiesChange = useCallback(
      async (attribute, newValue: IssueEntity[]) => {
        const value = data[attribute.name];

        const newOpportunitiesPromises = getNewOpportunities(newValue, value).map((opportunity) =>
          callService.mutate(id, {
            name: attribute.name,
            operation: 'add',
            value: opportunity,
          }),
        );
        const deletedOpportunitiesPromises = getDeletedOpportunities(newValue, value).map(
          (opportunity) =>
            callService.mutate(id, {
              name: attribute.name,
              operation: 'remove',
              value: opportunity,
            }),
        );

        return Promise.all([...newOpportunitiesPromises, ...deletedOpportunitiesPromises]);
      },
      [callService, data, id],
    );

    const handleCommonChange = useCallback(
      async (data: MutationData<PersonalCallData>) => {
        return callService.mutate(id, data);
      },
      [id, callService],
    );

    const handleDoneClick = async () => {
      const errorMessage = await callService.validateData(id);

      if (errorMessage) {
        userInteractionService.alert({
          title: i18nValidationAlert,
          description: errorMessage,
        });
      }
    };

    if (!callService.callData) {
      return null;
    }

    return (
      <Popup
        data-testid="call-popup"
        visible={isVisible}
        className={cx(css.CallPopup, {
          [css.CallPopup_isCallActive]: isCallActive,
        })}
      >
        <div className={css.CallPopup__body}>
          {isLoading ? (
            <Spin view="default" position="center" progress size="xxs" />
          ) : (
            <>
              <div className={css.CallPopup__header}>
                <div className={css.CallPopup__contact}>
                  <Text typography="body-short-l" weight="medium">
                    {i18nTitle}
                  </Text>
                  <Text typography="body-short-m" color="secondary" weight="medium">
                    {data.phoneNumber}
                  </Text>
                </div>
                <div className={css.CallPopup__right}>
                  <div className={css.CallPopup__timer}>
                    <Timer
                      startDate={data.startTime}
                      isActive={isCallActive}
                      timePassedMs={data.duration}
                    />
                  </div>
                  <Button
                    data-testid="call-popup-minimize-button"
                    view="clear"
                    onClick={onMinimize}
                    size="m"
                    icon={(className) => (
                      <Icon
                        svgSize="ml"
                        svg="minimize2"
                        className={cx(className, css.CallPopup__collapseIcon)}
                      />
                    )}
                  />
                </div>
              </div>
              <div className={css.CallPopup__content}>
                <CallAttributes
                  callData={callData}
                  onCommonChange={handleCommonChange}
                  onOpportunitiesChange={handleOpportunitiesChange}
                  onTagsChange={handleTagsChange}
                />
              </div>
              <div className={css.CallPopup__footer}>
                {!isCallActive && (
                  <Button
                    data-testid="call-popup-done-button"
                    view="action"
                    onClick={handleDoneClick}
                    size="m"
                  >
                    {i18nDone}
                  </Button>
                )}
              </div>
            </>
          )}
        </div>
      </Popup>
    );
  },
);
