import React, { useCallback, useContext, useEffect, useMemo, useState } from 'react';
import { observer } from 'mobx-react-lite';
import { XivaBackendEvents, XivaBackendEventType, XivaContext } from 'modules/xiva';
import { usePersonalCallService } from 'services/PersonalCallService';

import { CallStatus, isCallActivityData } from 'modules/xiva/types/PersonalCall';
import { CallPopupIcon } from '../CallPopupIcon';
import { CallPopup } from '../CallPopup';

export const CallPopupContainer = observer((() => {
  const callService = usePersonalCallService();

  const xiva = useContext(XivaContext);

  const { isCallDataLoading } = callService;

  const [lastActivityId, setLastActivityId] = useState<number | null>(null);
  const [isCallActive, setIsCallActive] = useState(Boolean(callService.callData?.data?.status));
  const [isVisible, setIsVisible] = useState(false);

  const preparePopup = (activityId: number) => {
    if (activityId) {
      setLastActivityId(activityId);
      setIsVisible(true);
    }
  };

  const handleCallStart = useCallback(
    (activityId: number) => {
      if (!activityId || lastActivityId === activityId) {
        return;
      }

      setIsCallActive(true);
      preparePopup(activityId);

      callService.loadCallData(activityId);
    },
    [callService, lastActivityId],
  );

  const handleCallEnd = useCallback(
    (activityId: number) => {
      if (activityId !== lastActivityId) {
        return;
      }

      setIsCallActive(false);

      if (!isVisible) {
        setIsVisible(true);
      }
    },
    [isVisible, lastActivityId],
  );

  const handleCallUpdate = useCallback(
    (activityId: number) => {
      if (activityId !== lastActivityId) {
        return;
      }

      if (callService && lastActivityId) {
        callService.reloadCallData(lastActivityId);
      }
    },
    [lastActivityId, callService],
  );

  const handleMinimize = useCallback(() => {
    setIsVisible(false);
  }, []);

  const handleToggle = useCallback(() => {
    setIsVisible((visible) => !visible);
  }, []);

  const activityToHandler = useMemo(
    () =>
      ({
        callOperatorAnswered: handleCallStart,
        callFinished: handleCallEnd,
        update: handleCallUpdate,
      } as Record<CallStatus, (id: number) => void>),
    [handleCallEnd, handleCallStart, handleCallUpdate],
  );

  const handleNewActivity = useCallback(
    (event: XivaBackendEvents[XivaBackendEventType.Activities]) => {
      if (isCallActivityData(event.detail)) {
        activityToHandler[event.detail.status]?.(event.detail.id);
      }
    },
    [activityToHandler],
  );

  useEffect(() => {
    xiva.addEventListener(XivaBackendEventType.Activities, handleNewActivity);
    return () => {
      xiva.removeEventListener(XivaBackendEventType.Activities, handleNewActivity);
    };
  }, [handleNewActivity, xiva]);

  useEffect(() => {
    if (lastActivityId) {
      return;
    }

    const loadLastActivity = async () => {
      const lastActivity = await callService.getLastNotDoneActivity();
      preparePopup(lastActivity?.id);
    };

    loadLastActivity();
  }, [callService, lastActivityId]);

  useEffect(() => {
    if (callService.callData?.data?.status === 'InProgress') {
      setIsCallActive(true);
      return;
    }

    if (callService.callData?.data?.status === 'Ended') {
      setIsCallActive(false);
      return;
    }
  }, [callService.callData?.data?.status]);

  if (!lastActivityId || callService.callData?.data?.isDone) {
    return null;
  }

  return (
    <>
      <CallPopupIcon isCallActive={isCallActive} onClick={handleToggle} />
      <CallPopup
        isCallActive={isCallActive}
        isVisible={isVisible}
        onMinimize={handleMinimize}
        isLoading={isCallDataLoading}
      />
    </>
  );
}) as React.FC);
