import React, { useContext, useEffect, useState } from 'react';
import { observer } from 'mobx-react-lite';
import { XivaBackendEvents, XivaBackendEventType, XivaContext } from 'modules/xiva';

import { usePersonalCallService } from 'services/PersonalCallService';
import { isCallActivityData } from 'modules/xiva/types/PersonalCall';
import CallToast from './CallToast/CallToast';

export const CallToastContainer = observer((() => {
  const callService = usePersonalCallService();
  const xiva = useContext(XivaContext);

  const { isCallDataLoading, callData } = callService;

  const [activityId, setActivityId] = useState<number | null>(null);

  useEffect(() => {
    const hideToast = () => {
      setActivityId(null);
    };

    const showToast = (callId: number) => {
      setActivityId(callId);
      callService.loadCallData(callId);
    };

    const handleNewActivity = (event: XivaBackendEvents[XivaBackendEventType.Activities]) => {
      if (isCallActivityData(event.detail)) {
        switch (event.detail.status) {
          case 'callOperatorAnswered':
          case 'callOperatorRejected': {
            hideToast();
            return;
          }
          case 'callStarted': {
            showToast(event.detail.id);
            return;
          }
        }
      }
    };

    xiva.addEventListener(XivaBackendEventType.Activities, handleNewActivity);
    return () => {
      xiva.removeEventListener(XivaBackendEventType.Activities, handleNewActivity);
    };
  }, [callService, xiva]);

  if (!activityId || isCallDataLoading) {
    return null;
  }

  return <CallToast callData={callData?.data} />;
}) as React.FC);
