import * as React from 'react';
import cx from 'classnames';
import LinkModule from 'components/LinkModule';
import Icon from 'lego/components/Icon';
import ToolTip from 'lego/components/ToolTip';
import withArray from 'components/PreviewValue/withArray';
import css from './ChangesRequest.module.css';

const getId = item => {
  if (typeof item === 'object') {
    return item.id;
  }

  return item;
};

const ArrayRender = withArray({ getId })();

export enum Operation {
  Add = 'Add',
  Remove = 'Remove',
}

interface DiffItem {
  value: any;
  operation?: Operation;
}

interface Request {
  issue: {
    id: number;
  };
  diff: DiffItem[];
  error?: boolean;
}

interface PropsBase {
  previewComponent: React.FunctionComponent<{ value: any }> | React.ComponentClass<{ value: any }>;
  previewComponentProps?: any;
}

interface Props extends PropsBase {
  items?: Request[];
}

interface DiffProps extends PropsBase {
  value: DiffItem;
}

const DiffItem = (props: DiffProps) => {
  const { value, previewComponent: PreviewComponent, previewComponentProps } = props;

  if (!value) {
    return null;
  }

  const { operation, value: data } = value;

  return (
    <span className={css[`b__item_operation_${operation}`]}>
      <PreviewComponent value={data} {...previewComponentProps} />
    </span>
  );
};

const ChangesRequest = (props: Props) => {
  const { items, previewComponent, previewComponentProps } = props;

  if (!Array.isArray(items)) {
    return null;
  }

  return (
    <div className={css.b}>
      <div className={css.b__header}>На утверждении</div>
      <div className={css.b__items}>
        {items.map(({ issue, diff, error }) => (
          <div className={cx(css.b__item, { [css.b__item_error]: error })}>
            <div>
              {issue && issue.id && (
                <LinkModule type={LinkModule.backendTypes.ISSUE} id={issue.id}>
                  {issue.id}
                </LinkModule>
              )}
              {error && (
                <ToolTip tooltip="Напишите на crm-bugs@yandex-team" padding="xs">
                  <Icon
                    pack={Icon.PACKS.MATERIAL}
                    icon="report"
                    className={css.b__itemReportIcon}
                  />
                </ToolTip>
              )}
            </div>
            {Array.isArray(diff) && !!diff.length && (
              <div>
                <ArrayRender
                  items={diff}
                  component={DiffItem}
                  props={{ previewComponent, previewComponentProps }}
                />
              </div>
            )}
          </div>
        ))}
      </div>
    </div>
  );
};

ChangesRequest.defaultProps = {
  previewComponent: () => null,
};

export default ChangesRequest;
