import React from 'react';
import cx from 'classnames';
import isUndefined from 'lodash/isUndefined';
import ToolTip from 'lego/components/ToolTip';
import Link from '@crm/components/dist/lego2/Link';
import { accountLink } from 'pages/accounts/utils/linkCreator';
import css from './ClientExternalLinks.module.css';
import SpaceSvg from './svg/space.react.svg';
import DirectSvg from './svg/direct.react.svg';
import MetrikaSvg from './svg/metrika.react.svg';
import RsyaSvg from './svg/rsya.react.svg';
import SpravochnikSvg from './svg/spravochnik.react.svg';
import MarketSvg from './svg/market.react.svg';
import ZenSvg from './svg/zen.react.svg';
import AudienceSvg from './svg/audience.react.svg';
import BalanceSvg from './svg/balance.react.svg';
import MediasaleSvg from './svg/mediasale.react.svg';
import TelephonySvg from './svg/telephony.react.svg';
import { Client } from './ClientExternalLinks.types';

const lg = {
  crm: 'CRM Space',
  balance: 'Баланс',
  market: 'Маркет',
  mediasel: 'Дисплей',
  metrika: 'Метрика',
  sprav: 'Справочник',
  direct: 'Директ',
  audience: 'Я.Аудитория',
  rsya: 'РСЯ',
  zen: 'Дзен',
  yaTelephony: 'Я.Телефония',
};

const LinkBuilder = {
  balance(clientId: number) {
    return `https://admin.balance.yandex.ru/passports.xml?tcl_id=${clientId}`;
  },
  balanceYaTeam(clientId: number) {
    return `https://admin.balance.yandex-team.ru/passports.xml?tcl_id=${clientId}`;
  },
  direct(login: string, isAgency: boolean = false) {
    return isAgency
      ? `https://direct.yandex.ru/registered/main.pl?cmd=showClients&ulogin=${login}`
      : `https://direct.yandex.ru/registered/main.pl?cmd=showCamps&ulogin=${login}`;
  },
  market(clientId: number) {
    return `https://partner.market.yandex.ru/?client_id=${clientId}`;
  },
  sprav(login: string) {
    return `https://yandex.ru/sprav/orders?auth=${login}`;
  },
  metrika(login: string, isAgency: boolean = false) {
    return isAgency
      ? 'https://metrika.yandex.ru/admin/search-counters/'
      : `https://metrika.yandex.ru/list?ulogin=${login}`;
  },
  mediasel(clientId, isAgency = false) {
    return isAgency
      ? 'https://display.yandex.ru/campaigns/'
      : `https://display.yandex.ru/advertisers/${clientId}`;
  },
  zen: (publisherId: string | number) => `https://zen.yandex.ru/id/${publisherId}`,
  audience: (login: string) => `https://audience.yandex.ru/?ulogin=${login}`,
  rsya: (login: string) =>
    `https://partner2.yandex.ru/v2/sites/?search=%7B"login.user.login"%3A%5B"LIKE"%2C"${login}"%5D%7D`,
  telephony: (id: string) => `https://telephony.yandex.ru/admin/clients/${id}`,
};

const getExternalLink = (
  href: string,
  type: string,
  disabled: boolean = false,
  Component: string | React.ComponentType<React.SVGAttributes<HTMLOrSVGElement>> = 'i',
  options: {
    tabIndex?: number;
  },
) =>
  disabled ? (
    <Component className={cx(css.b__icon, css.b__icon_disabled)} />
  ) : (
    <a
      href={href}
      className={css.b__link}
      title={lg && lg[type] ? lg[type] : ''}
      target="_blank"
      rel="noopener noreferrer"
      tabIndex={options.tabIndex}
    >
      <Component className={css.b__icon} />
    </a>
  );

interface Props {
  className?: string;
  client?: Client;
  size?: string;
  tabIndex?: number;
  withLineHeight?: boolean;
}

const isArray1Length = function<T>(array: T[]): array is [T] {
  return array.length === 1;
};

type RenderZen = (data: {
  zenPublisherIds: (string | number)[];
  disable: boolean;
  tabIndex?: number;
}) => React.ReactNode;
const renderZen: RenderZen = ({ zenPublisherIds, disable, tabIndex }) => {
  if (!isArray1Length(zenPublisherIds)) {
    return (
      <ToolTip
        autoZIndex
        padding="xs"
        hideDelay={300}
        tooltip={
          <div className={css.tooltip}>
            {zenPublisherIds.map((id) => (
              <Link
                key={id}
                className={css.tooltip__link}
                href={LinkBuilder.zen(id)}
                theme="normal"
              >
                {id}
              </Link>
            ))}
          </div>
        }
      >
        <ZenSvg className={css.b__icon} />
      </ToolTip>
    );
  }

  return getExternalLink(LinkBuilder.zen(zenPublisherIds[0]), 'zen', disable, ZenSvg, {
    tabIndex,
  });
};

const ClientExternalLinks: React.FC<Props> = (props) => {
  const { className, client, size = '', withLineHeight = false, tabIndex } = props;

  if (client == null || !client.services) {
    return null;
  }

  const cId = client.info && client.info.clientId;
  const cLogin = client.info && client.info.login;
  const cServices = client.services;
  const cType = client.info && client.info.type;
  const telephonyId = client.info && client.info.yatelMcId;
  const { zenPublisherIds } = client.info;
  const isAgency = cType === 'Agency';

  return (
    <span
      className={cx(
        css.b,
        css[`b_size_${size}`],
        { [css.b_withLineHeight]: withLineHeight },
        className,
      )}
    >
      {client.info &&
        client.info.id &&
        getExternalLink(accountLink(client.info.id, true), 'crm', false, SpaceSvg, {
          tabIndex,
        })}
      {!isUndefined(cServices.zen) &&
        Array.isArray(zenPublisherIds) &&
        zenPublisherIds.length &&
        renderZen({
          zenPublisherIds,
          disable: !cServices.zen,
          tabIndex,
        })}
      {!isUndefined(cServices.balance) &&
        getExternalLink(LinkBuilder.balance(cId), 'balance', !cServices.balance, BalanceSvg, {
          tabIndex,
        })}
      {!isUndefined(cServices.balanceYaTeam) &&
        getExternalLink(
          LinkBuilder.balanceYaTeam(cId),
          'balance',
          !cServices.balanceYaTeam,
          BalanceSvg,
          {
            tabIndex,
          },
        )}
      {!isUndefined(cServices.direct) &&
        getExternalLink(
          LinkBuilder.direct(cLogin, isAgency),
          'direct',
          !cServices.direct,
          DirectSvg,
          {
            tabIndex,
          },
        )}
      {!isUndefined(cServices.market) &&
        getExternalLink(LinkBuilder.market(cId), 'market', !cServices.market, MarketSvg, {
          tabIndex,
        })}
      {!isUndefined(cServices.mediasel) &&
        getExternalLink(
          LinkBuilder.mediasel(cId, isAgency),
          'mediasel',
          !cServices.mediasel,
          MediasaleSvg,
          {
            tabIndex,
          },
        )}
      {!isUndefined(cServices.metrika) &&
        getExternalLink(
          LinkBuilder.metrika(cLogin, isAgency),
          'metrika',
          !cServices.metrika,
          MetrikaSvg,
          {
            tabIndex,
          },
        )}
      {!isUndefined(cServices.sprav) &&
        getExternalLink(LinkBuilder.sprav(cLogin), 'sprav', !cServices.sprav, SpravochnikSvg, {
          tabIndex,
        })}
      {!isUndefined(cServices.audience) &&
        getExternalLink(
          LinkBuilder.audience(cLogin),
          'audience',
          !cServices.audience,
          AudienceSvg,
          {
            tabIndex,
          },
        )}
      {!isUndefined(cServices.rsya) &&
        getExternalLink(LinkBuilder.rsya(cLogin), 'rsya', !cServices.rsya, RsyaSvg, {
          tabIndex,
        })}
      {!isUndefined(cServices.yaTelephony) &&
        getExternalLink(
          LinkBuilder.telephony(telephonyId),
          'yaTelephony',
          !cServices.yaTelephony,
          TelephonySvg,
          {
            tabIndex,
          },
        )}
    </span>
  );
};

export default ClientExternalLinks;
