import React from 'react';
import { FormattedDate } from 'react-intl';
import { ContactData, Email } from 'types';
import WordBreak from 'components/WordBreak';
import Button from '@crm/components/dist/lego2/Button';
import ToolTip from 'lego/components/ToolTip';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import Icon2 from '@crm/components/dist/lego2/Icon';
import Icon from 'lego/components/Icon';
import NewMailModal from 'pages/mail/components/NewMailModal';
import * as keyset from './Contact.i18n';
import { Phone, PhoneProps } from './Phone';
import css from './ContactFull.module.css';
import Comment from './Comment';

const i18n = createI18N(keyset);
const i18nForBroadcast = i18n('forBroadcast');
const i18nWrite = i18n('write');
const i18nUpdated = i18n('updated');

interface RenderMailData {
  email: Email;
  contactId: number;
  accountId: number;
  canWriteMail?: boolean;
}

const renderEmail = (data: RenderMailData) => {
  const { email, contactId, accountId, canWriteMail } = data;

  return (
    <div key={email.email} className={css.emailItem}>
      {email.isBroadcast && (
        <i title={i18nForBroadcast} className="material-icons">
          mail
        </i>
      )}
      {email.email || '–'}
      {canWriteMail && email.email && email.email !== '–' && (
        <NewMailModal
          name={`NEW_MAIL_CLIENT:${contactId}:${email.id}`}
          loadArgs={{ accountId, email: email.email }}
          loadOnMount
        >
          {(modal, open) => (
            <div className={css.action}>
              <a tabIndex={0} onClick={open}>
                {i18nWrite}
              </a>
              {modal}
            </div>
          )}
        </NewMailModal>
      )}
    </div>
  );
};

interface Props {
  className?: string;
  contact: ContactData;
  id: number;
  onEdit: () => void;
  onDelete: () => void;
  accountId: number;
  canEdit?: boolean;
  canWriteMail?: boolean;
  isCompactView?: boolean;
  withEmptyActionsPlaceholder?: boolean;
  callToComponent?: PhoneProps['callToComponent'];
}

const iconEdit = (iconCls) => <Icon2 svg="pencil" className={iconCls} />;

const iconDelete = (iconCls) => <Icon2 svg="mainToolbarDelete" className={iconCls} />;

const ContactFull = (props: Props): null | React.ReactElement => {
  const {
    className,
    id,
    contact,
    onEdit,
    onDelete,
    accountId,
    canEdit,
    canWriteMail,
    isCompactView,
    withEmptyActionsPlaceholder,
    callToComponent,
  } = props;

  if (!contact) {
    return null;
  }

  const {
    isMain,
    comment,
    firstName,
    lastName,
    middleName,
    position,
    emails,
    phones,
    source,
    language,
    modifiedOn,
  } = contact;

  let lang: string | null = null;
  if (language) {
    lang = language.code;
  }

  let job: string | null = null;
  if (position) {
    job = position.name;
  }

  const popup = modifiedOn ? (
    <span>
      {i18nUpdated}&nbsp;
      <FormattedDate value={new Date(modifiedOn)} year="numeric" month="long" day="numeric" />
    </span>
  ) : null;

  return (
    <WordBreak className={cx(className, css.contact, { [css.contact_compact]: isCompactView })}>
      <div className={css.row}>
        <div className={css.name}>
          {isMain && (
            <Icon pack={Icon.PACKS.MATERIAL} icon="star" className={css.contact__mainIcon} />
          )}
          {!lastName && !firstName && !middleName && '–'}
          {lastName} {firstName} {middleName} {lang ? `(${lang})` : null}
          {<div className={css.position}>{job || '–'}</div>}
          {
            <div className={css.extra}>
              <ToolTip tooltip={popup} stopPropagation padding="s">
                <span title={i18n('source', { source: source || '-' })}>
                  <i className="material-icons">system_update_alt</i>
                  {source || '–'}
                </span>
              </ToolTip>
            </div>
          }
        </div>
        <div className={css.email}>
          {emails &&
            emails.length > 0 &&
            emails.map((email) =>
              renderEmail({
                email,
                contactId: id,
                accountId,
                canWriteMail,
              }),
            )}
          {(!emails || !emails.length) && '–'}
        </div>
        {
          <div
            className={cx(css.phone, {
              [css.phone_actionPlaceholder]: withEmptyActionsPlaceholder,
            })}
          >
            {phones &&
              phones.length > 0 &&
              phones.map((phone) => (
                <Phone key={phone.id} phone={phone} callToComponent={callToComponent} />
              ))}
            {(!phones || !phones.length) && '–'}
          </div>
        }
        {canEdit && (
          <div className={css.actions}>
            <Button view="pseudo" size="xs" onClick={onEdit} icon={iconEdit} />
            <Button view="danger" size="xs" onClick={onDelete} icon={iconDelete} />
          </div>
        )}
      </div>
      {comment && <Comment data={comment} className={css.comment} />}
    </WordBreak>
  );
};

ContactFull.defaultProps = {
  className: undefined,
  comment: undefined,
  accountId: undefined,
  canEdit: undefined,
  onEdit: undefined,
  onDelete: undefined,
};

export default ContactFull;
