import React, { ComponentType } from 'react';
import { ContactData } from 'types';
import { useResizeNotify } from 'utils/hooks/useResizeNotify';
import createI18N from '@yandex-int/i18n';
import * as keyset from './Contact.i18n';
import css from './Contacts.module.css';
import ContactFull from './ContactFull';
import { PhoneProps } from './Phone';
import cssContact from './ContactFull.module.css';
import { ContactComponentProps } from './Contacts.types';

const i18n = createI18N(keyset);
const i18nNoContacts = i18n('noContacts');
const i18nFullName = i18n('fullName');
const i18nEmail = i18n('email');
const i18nPhoneNumber = i18n('phoneNumber');

interface ContactsProps {
  canWriteMail?: boolean;
  contacts?: ContactData[];
  callToComponent?: PhoneProps['callToComponent'];
  contactComponent?: ComponentType<ContactComponentProps>;
  canEdit?: boolean;
}

const Contacts = (props: ContactsProps) => {
  const {
    canWriteMail,
    contacts,
    callToComponent,
    contactComponent: ContactComponent = ContactFull,
    canEdit,
  } = props;

  const [domNode, width] = useResizeNotify<HTMLDivElement>();

  if (!(Array.isArray(contacts) && contacts.length)) {
    return <span>{i18nNoContacts}</span>;
  }

  const isCompactView = Number(width) < 300;

  return (
    <div ref={domNode} className={css.b} hidden={width == null}>
      {!isCompactView && (
        <div className={css.b__header}>
          <div className={cssContact.name}>{i18nFullName}</div>
          <div className={cssContact.email}>{i18nEmail}</div>
          <div className={cssContact.phone}>{i18nPhoneNumber}</div>
        </div>
      )}
      <div>
        {contacts.map((contact) => (
          <ContactComponent
            className={css.b__contact}
            key={contact.contactId}
            id={contact.contactId}
            contact={contact}
            canEdit={canEdit}
            canWriteMail={canWriteMail}
            isCompactView={isCompactView}
            callToComponent={callToComponent}
          />
        ))}
      </div>
    </div>
  );
};

export default Contacts;
