import React, { useState, useEffect, FC, RefObject } from 'react';
import { DatePicker } from 'components/DatePicker';
import { useIMask } from '@crm/react-imask';
import cx from 'classnames';
import Textinput from '@crm/components/dist/lego2/Textinput';
import { Popup } from '@crm/components/dist/Popup';
import { DateInputProps } from './DateInput.types';
import { formatDate } from './DateInput.utils';
import css from './DateInput.module.css';

export const DateInput: FC<DateInputProps> = ({ onChange, value }) => {
  const [calendarVisibility, setCalendarVisibility] = useState(false);
  const [inputValue, setInputValue] = useState<string>(() => formatDate(value));

  const { ref, maskRef } = useIMask({
    mask: Date,
  });
  const mask = maskRef.current;
  const selectedDate = value || null;

  useEffect(() => {
    setInputValue(formatDate(value));
    mask?.updateValue();
  }, [value, mask]);

  const selectDateHandler = (date: Date) => {
    onChange?.(date);
    setCalendarVisibility(false);
  };

  const changeInputHandler = (str: string) => {
    setInputValue(str);
    mask?.updateValue?.();
    if (str === '') {
      onChange?.(undefined);
      setCalendarVisibility(true);
    }
    if (mask?.masked?.isComplete) {
      selectDateHandler(mask.typedValue);
    }
  };

  const blurHandler = () => {
    if (inputValue === '' && value) {
      onChange?.(undefined);
      return;
    }
    if (mask && !mask.masked.isComplete) {
      setInputValue(formatDate(value));
      return;
    }
  };

  const focusHandler = () => {
    mask?.updateValue();
    setCalendarVisibility(true);
  };

  const closePopupHandler = () => {
    mask?.updateValue();
    setCalendarVisibility(false);
  };

  const keyDownHandler = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      if (!selectedDate) {
        const currentDate = new Date();
        currentDate.setHours(0);
        currentDate.setMinutes(0);
        currentDate.setSeconds(0);
        currentDate.setMilliseconds(0);
        selectDateHandler(currentDate);
        return;
      }
      setCalendarVisibility(false);
    }
  };

  return (
    <div className={css.DateInput} data-testid="DateInput">
      <div>
        <Textinput
          type="text"
          placeholder="дд.мм.гггг"
          className={cx(css.DateInput__input, {
            [css.DateInput__input_empty]: !inputValue,
          })}
          controlRef={ref as RefObject<HTMLInputElement>}
          value={inputValue}
          onChange={changeInputHandler}
          onBlur={blurHandler}
          onFocus={focusHandler}
          onKeyDown={keyDownHandler}
        />
        <Popup
          isVisible={calendarVisibility}
          anchor={ref.current}
          className={css.DateInput__popup}
          direction="bottom-start"
          onClose={closePopupHandler}
        >
          <DatePicker
            view="mg"
            selected={selectedDate}
            onSelect={selectDateHandler}
            disabledKeyboardNavigation
            fixedHeight
          />
        </Popup>
      </div>
    </div>
  );
};
