import React, { useState, useRef, FC } from 'react';
import cx from 'classnames';
import { isToday } from 'date-fns';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import Popup from '@crm/components/dist/lego2/Popup';
import { DatePicker } from 'components/DatePicker';
import { formatDate } from './DateNavigator.utils';
import { DateNavigatorProps } from './DateNavigator.types';
import css from './DateNavigator.module.css';

export const DateNavigator: FC<DateNavigatorProps> = (props) => {
  const { className, value, onChange } = props;
  const [isEditing, setIsEditing] = useState(false);
  const buttonRef = useRef<HTMLButtonElement>(null);

  const onPrevDateClick = () => {
    const prevDate = new Date(value);
    prevDate.setDate(prevDate.getDate() - 1);
    onChange?.(prevDate);
  };

  const onNextDateClick = () => {
    const nextDate = new Date(value);
    nextDate.setDate(nextDate.getDate() + 1);
    onChange?.(nextDate);
  };

  const onTodayClick = () => {
    onChange?.(new Date());
  };

  const onDateClick = () => {
    setIsEditing((val) => !val);
  };

  const onDateChange = (date: Date) => {
    onChange?.(date);
    setIsEditing(false);
  };

  const onDatePickerClose = () => {
    setIsEditing(false);
  };

  return (
    <div className={cx(css.DateNavigator, className)}>
      <div className={css.DateNavigator__date}>
        <Button
          pin="round-brick"
          data-testid="prev-date-button"
          onClick={onPrevDateClick}
          icon={(cls) => (
            <Icon
              className={cx(cls, css.DateNavigator__prevDateIcon)}
              svg="arrowShort"
              svgSize="ms"
            />
          )}
        />

        <Button
          onClick={onDateClick}
          ref={buttonRef}
          data-testid="selected-date-button"
          iconLeft={(cls) => <Icon className={cls} svg="event" svgSize="ml" />}
          pin="brick-brick"
        >
          {formatDate(value)}
        </Button>

        <Button
          pin="brick-round"
          data-testid="next-date-button"
          onClick={onNextDateClick}
          icon={(cls) => (
            <Icon
              className={cx(cls, css.DateNavigator__nextDateIcon)}
              svg="arrowShort"
              svgSize="ms"
            />
          )}
        />

        <Popup
          visible={isEditing}
          anchor={buttonRef}
          target="anchor"
          direction="bottom-start"
          className={css.DateNavigator__popup}
          onClose={onDatePickerClose}
        >
          <DatePicker
            view="mg"
            selected={value}
            onSelect={onDateChange}
            disabledKeyboardNavigation
            fixedHeight
          />
        </Popup>
      </div>

      <Button onClick={onTodayClick} disabled={isToday(value)}>
        Сегодня
      </Button>
    </div>
  );
};
