import React, { useState, FC, useMemo, useEffect } from 'react';
import { DateInput } from 'components/DateInput';
import { TimeInput, getHoursAndMinutes, formatTime } from 'components/TimeInput';
import toLocalISOString from 'utils/date/toLocalISOString';
import { DateTimeInputProps } from './DateTimeInput.types';
import { isTimeValid } from './DateTimeInput.utils';

import css from './DateTimeInput.module.css';

export const DateTimeInput: FC<DateTimeInputProps> = ({ onChange, value }) => {
  const dateTime = useMemo(() => (value ? new Date(value) : undefined), [value]);
  const [time, setTime] = useState('');

  useEffect(() => {
    const newTime = formatTime(dateTime);
    setTime(newTime);
  }, [dateTime]);

  const selectDate = (date: Date) => {
    if (time) {
      date.setMinutes(getHoursAndMinutes(time).minute);
      date.setHours(getHoursAndMinutes(time).hour);
    }

    onChange?.(toLocalISOString(date));
  };

  const changeDateHandler = (date?: Date) => {
    if (!date) {
      onChange?.(undefined);
      return;
    }
    selectDate(date);
  };

  const changeTimeHandler = (newTime: string) => {
    setTime(newTime);
    if (!dateTime || (newTime && !isTimeValid(newTime))) {
      return;
    }
    const newDate = new Date(dateTime);
    newDate.setMinutes(getHoursAndMinutes(newTime).minute);
    newDate.setHours(getHoursAndMinutes(newTime).hour);
    onChange?.(toLocalISOString(newDate));
  };

  return (
    <div className={css.DateTimeInput}>
      <div>
        <DateInput onChange={changeDateHandler} value={dateTime} />
      </div>
      <div>
        <TimeInput value={time} onChange={changeTimeHandler} />
      </div>
    </div>
  );
};
