import { makeAutoObservable } from 'mobx';
import Bluebird from 'bluebird';
import { Form } from 'types/api/form/Form';
import { AnyObject } from 'react-final-form';
import { get, jsonApiCall } from 'api/common';
import Access from 'utils/Access';
import { formatFormValuesToDataSet, getInitialValues } from 'components/FormByScheme';
import { ChangedField } from 'components/FinalForm/Form/Form.context';

type FunctionWithAnyArgs = ((...args: unknown[]) => void) | undefined;

export class DynamicallyModalFormService {
  public loadError?: Error;
  public formData?: Form;
  public initialValues: AnyObject;
  public isUpdatingFields: boolean;
  public canEditForm = false;
  public isLoading = false;
  public formClosingListener;
  public successSubmitListener;
  private apiRequestPromise?: Bluebird<unknown>;

  constructor(private url?: string) {
    if (url) {
      this.url = url;
    }
    makeAutoObservable(this, undefined, { autoBind: true });
  }

  public setUrl(url: string) {
    this.url = url;
    return this;
  }

  public setLoading(value: boolean) {
    this.isLoading = value;
  }

  public setFormClosingListener(fn: FunctionWithAnyArgs) {
    this.formClosingListener = fn;
  }

  public setSuccessSubmitListener(fn: FunctionWithAnyArgs) {
    this.successSubmitListener = fn;
  }

  public resetAll() {
    this.url = undefined;
    this.formData = undefined;
    this.canEditForm = false;
    this.setFormClosingListener(undefined);
    this.setSuccessSubmitListener(undefined);
  }

  public handleCloseForm() {
    if (this.formClosingListener) {
      this.formClosingListener();
    }
    if (this.apiRequestPromise) {
      this.apiRequestPromise.cancel();
    }
    this.resetAll();
  }

  public handleSuccessSubmit(response) {
    if (this.successSubmitListener) {
      this.successSubmitListener(response);
    }
    this.resetAll();
  }

  public async loadForm() {
    this.setLoading(true);
    if (this.url) {
      this.apiRequestPromise = get<Form>({ url: this.url, global: false })
        .then((data) => {
          this.setFormData(data);
          return data;
        })
        .catch((error) => {
          this.setLoadError(error);
          throw error;
        })
        .finally(() => {
          this.setLoading(false);
        });
      await this.apiRequestPromise;
    }
  }

  public async onChange(changed: ChangedField, oldValues: AnyObject) {
    const fieldMeta = this.formData?.meta.fields.find(({ type, id }) => type + id === changed.name);

    if (fieldMeta?.isFieldsUpdateNeeded) {
      const values = { ...oldValues, [changed.name]: changed.value };
      await this.updateFields(values, oldValues);
    }
  }

  private async updateFields(values: AnyObject, oldValues: AnyObject) {
    const editingRowId = this.formData?.data[0]?.id;
    const fields = formatFormValuesToDataSet(values, this.formData, editingRowId);
    const oldFields = formatFormValuesToDataSet(oldValues, this.formData, editingRowId);

    if (!this.url) return;
    const [path, query] = this.url.split('?');
    const url = path + '/update-fields' + (query ? '?' + query : '');

    if (this.apiRequestPromise) {
      this.apiRequestPromise.cancel();
    }

    this.isUpdatingFields = true;
    const timeout = setTimeout(() => {
      this.setLoading(true);
    }, 300);

    this.apiRequestPromise = jsonApiCall({
      url,
      data: {
        id: editingRowId,
        oldFields: oldFields.fields,
        fields: fields.fields,
      },
      global: false,
    })
      .then(this.setFormData)
      .finally(() => {
        clearTimeout(timeout);
        this.setLoading(false);
        this.isUpdatingFields = false;
      });

    await this.apiRequestPromise;
  }

  public async submitForm(values: AnyObject) {
    const editingRowId = this.formData?.data[0]?.id;
    const row = formatFormValuesToDataSet(values, this.formData, editingRowId);
    if (this.url) {
      this.apiRequestPromise = jsonApiCall({
        url: this.url,
        data: { args: row },
        global: false,
      }).then(({ results }) => {
        this.handleSuccessSubmit(results);
      });
      await this.apiRequestPromise;
    }
  }

  private setLoadError(error: Error) {
    this.loadError = error;
  }

  private setFormData(data: Form) {
    this.initialValues = getInitialValues(data);
    this.formData = data;
    this.canEditForm = data.meta.fields.reduce(
      (ac, field) => ac || Access.isEdit(field.access),
      false,
    );
  }
}
